package com.atlassian.bamboo.plugins.tomcat.configuration;

import com.atlassian.bamboo.collections.ActionParametersMap;
import com.atlassian.bamboo.security.EncryptionService;
import com.atlassian.bamboo.task.AbstractTaskConfigurator;
import com.atlassian.bamboo.task.TaskDefinition;
import com.atlassian.bamboo.utils.error.ErrorCollection;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.opensymphony.xwork.TextProvider;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.Set;

public abstract class AbstractTomcatConfigurator extends AbstractTaskConfigurator
{
    private static final Logger log = Logger.getLogger(AbstractTomcatConfigurator.class);
    // ------------------------------------------------------------------------------------------------------- Constants
    public static final String TOMCAT_URL = "tomcatUrl";
    public static final String TOMCAT_USERNAME = "tomcatUsername";
    public static final String TOMCAT_PLAIN_PASSWORD = "tomcatPassword";
    public static final String TOMCAT_PASSWORD = "encTomcatPassword";


    public static final String TOMCAT_6 = "tomcat6";

    public static final String APP_CONTEXT = "appContext";

    private static final Set<String> FIELDS_TO_COPY = ImmutableSet.<String>builder().add(TOMCAT_URL,
                                                                                         TOMCAT_USERNAME,
                                                                                         TOMCAT_PASSWORD,
                                                                                         APP_CONTEXT,
                                                                                         TOMCAT_6).build();
    private static final String PASSWORD_CHANGE = "passwordChange";

    // ------------------------------------------------------------------------------------------------- Type Properties
    // ---------------------------------------------------------------------------------------------------- Dependencies

    protected TextProvider textProvider;
    private EncryptionService encryptionService;

    // ---------------------------------------------------------------------------------------------------- Constructors
    // ----------------------------------------------------------------------------------------------- Interface Methods

        @Override
    public void populateContextForCreate(@NotNull final Map<String, Object> context)
    {
        super.populateContextForCreate(context);
        context.put(TOMCAT_URL, "http://localhost:8080/manager/");
        context.put(TOMCAT_USERNAME, "admin");
        context.put(APP_CONTEXT, "/test");
    }

    @Override
    public void populateContextForEdit(@NotNull final Map<String, Object> context, @NotNull final TaskDefinition taskDefinition)
    {
        super.populateContextForEdit(context, taskDefinition);
        taskConfiguratorHelper.populateContextWithConfiguration(context, taskDefinition, Iterables.concat(FIELDS_TO_COPY, getFieldsToCopy()));
    }

    @Override
    public void populateContextForView(@NotNull final Map<String, Object> context, @NotNull final TaskDefinition taskDefinition)
    {
        super.populateContextForView(context, taskDefinition);
        taskConfiguratorHelper.populateContextWithConfiguration(context, taskDefinition, Iterables.concat(FIELDS_TO_COPY, getFieldsToCopy()));
    }

    @Override
    public void validate(@NotNull final ActionParametersMap params, @NotNull final ErrorCollection errorCollection)
    {
        super.validate(params, errorCollection);

        final String url = params.getString(TOMCAT_URL);
        if (StringUtils.isEmpty(url))
        {
            errorCollection.addError(TOMCAT_URL, textProvider.getText("tomcat.url.error"));
        }

        if (StringUtils.isEmpty(params.getString(TOMCAT_USERNAME)))
        {
            errorCollection.addError(TOMCAT_USERNAME, textProvider.getText("tomcat.username.error"));
        }

        if (StringUtils.isEmpty(params.getString(TOMCAT_PLAIN_PASSWORD)) && params.getBoolean(PASSWORD_CHANGE))
        {
            errorCollection.addError(TOMCAT_PLAIN_PASSWORD, textProvider.getText("tomcat.password.error"));
        }

        if (StringUtils.isEmpty(params.getString(APP_CONTEXT)))
        {
            errorCollection.addError(APP_CONTEXT, textProvider.getText("tomcat.context.error"));
        }
    }

    @NotNull
    @Override
    public Map<String, String> generateTaskConfigMap(@NotNull final ActionParametersMap params, @Nullable final TaskDefinition previousTaskDefinition)
    {
        final Map<String, String> config = super.generateTaskConfigMap(params, previousTaskDefinition);

        taskConfiguratorHelper.populateTaskConfigMapWithActionParameters(config, params, Iterables.concat(FIELDS_TO_COPY, getFieldsToCopy()));

        String passwordChange = params.getString(PASSWORD_CHANGE);
        if ("true".equals(passwordChange))
        {
            final String password = params.getString(TOMCAT_PLAIN_PASSWORD);
            config.put(TOMCAT_PASSWORD, encryptionService.encrypt(password));
        }
        else if (previousTaskDefinition != null)
        {
            config.put(TOMCAT_PASSWORD, previousTaskDefinition.getConfiguration().get(TOMCAT_PASSWORD));
        }
        else
        {
            final String password = params.getString(TOMCAT_PLAIN_PASSWORD);
            config.put(TOMCAT_PASSWORD, encryptionService.encrypt(password));
        }

        String tomcat6 = params.getString(TOMCAT_6);
        if ("true".equals(tomcat6))
        {
            config.put(TOMCAT_6, Boolean.TRUE.toString());
        }

        return config;
    }

    protected abstract Set<String> getFieldsToCopy();

    // -------------------------------------------------------------------------------------------------- Action Methods
    // -------------------------------------------------------------------------------------------------- Public Methods
    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators

    public void setTextProvider(final TextProvider textProvider)
    {
        this.textProvider = textProvider;
    }

    public void setEncryptionService(final EncryptionService encryptionService)
    {
        this.encryptionService = encryptionService;
    }
}
