define('branchinator/views/layout/messages-view', [
    'aui/flag',
    'backbone',
    'branchinator/utils/radio-utils',
    'underscore'
], function(
    Flag,
    Backbone,
    radioUtils,
    _
) {
    'use strict';

    /**
     * View for displaying all types of messages (info, warning, error, success) on branch status page. This view
     * listens to global events from Backbone.Radio.
     *
     * Parameters:
     * - hasRepositories - whether there are global, branchable repositories defined for this Bamboo instance
     *
     * Handled events:
     * - message:info    - displays an info message (message to display passed as parameter)
     * - message:warn    - displays a warning message (as above)
     * - message:error   - displays an error message (as above)
     * - message:success - displays a success message (as above)
     * - message:clear   - clears all messages
     *
     * Instead of passing the message text as parameter, it is possible to pass an object with additional properties.
     * The following settings are allowed:
     * - body  - the message content
     * - title - the title of the message
     * - close - whether the message should be closeable or not; the value should be either 'auto' (message will close
     *           automatically), 'manual' (message requires manual closing) or 'never' (message is permanent)
     * - clear - when set to true, other messages will be cleared before displaying this one
     */
    return Marionette.ItemView.extend({
        className: function() {
            return this.getOption('hasRepositories') ? 'hasRepositories' : '';
        },

        template: false,
        openedFlags: [],

        onShow: function() {
            radioUtils.listenTo('message:info', _.bind(this.showMessage, this, 'info'));
            radioUtils.listenTo('message:warn', _.bind(this.showMessage, this, 'warning'));
            radioUtils.listenTo('message:error', _.bind(this.showMessage, this, 'error'));
            radioUtils.listenTo('message:success', _.bind(this.showMessage, this, 'success'));
            radioUtils.listenTo('message:clear', this.clearMessage.bind(this));
        },

        showMessage: function(type, message) {
            var messageOpts = typeof message === 'object' ? message : { body: message };
            if (messageOpts.clear) {
                this.clearMessage();
            }

            var close = messageOpts.close || 'manual';
            if (close === 'never') {
                // permanent message will be attached to the DOM above all content on page
                var template = aui.message[type];
                this.$el.append(template({
                    titleContent: messageOpts.title || '',
                    content: messageOpts.body || ''
                }));
            } else {
                // temporary, closeable message will be displayed using aui-flag
                this.openedFlags.push(new Flag({
                    type: type,
                    close: close,
                    title: messageOpts.title || '',
                    body: messageOpts.body || ''
                }));
            }
        },

        clearMessage: function() {
            this.$el.empty();
            this.openedFlags.forEach(function(flag) {
                flag.close();
            });
            this.openedFlags = [];
        }
    });
});
