define('branchinator/views/layout/content/integrations/pull-requests-table-view', [
    'branchinator/behaviors/fetch-state-aware-behavior',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'branchinator/views/layout/content/integrations/pull-requests-table-row-view',
    'jquery',
    'marionette'
], function(
    FetchStateAwareBehavior,
    bnUtils,
    radioUtils,
    PullRequestsTableRowView,
    $,
    Marionette
) {
    'use strict';

    /**
     * Data for sending analytics event when user clicks "create pull request" link.
     */
    var createPullRequestAnalyticsData = JSON.stringify({
        name: 'branch.view.action.pullRequest.create.clicked'
    });

    var PullRequestsTableEmptyView = Marionette.ItemView.extend({
        tagName: 'tr',
        className: function() {
            return this.getOption('isFetchError') ? 'fetchError' : 'noResults';
        },

        template: function() {
            return bamboo.plugins.branchinator.renderPullRequestsTableEmptyView({
                fetched: this.getOption('isFetched'),
                isFetchError: this.getOption('isFetchError'),
                fetchErrorContent: this.getOption('fetchErrorContent')
            });
        }
    });

    /**
     * Displays table of pull requests for the currently selected branch.
     *
     * Parameters:
     * - model - branch selection model
     * - collection - pull requests collection
     *
     * TODO: after Marionette upgrade to 2.4.4+ migrate all 'onChildview...' handlers to 'childEvents' map
     */
    return Marionette.CompositeView.extend({

        childView: PullRequestsTableRowView,
        childViewContainer: 'tbody',

        emptyView: PullRequestsTableEmptyView,
        emptyViewOptions: function() {
            return {
                isFetchError: this.isFetchError,
                fetchErrorContent: this.fetchErrorContent,
                isFetched: this.isFetched()
            };
        },

        collectionEvents: {
            'sync': 'onCollectionSync',
            'error': 'onCollectionError'
        },

        behaviors: {
            FetchStateAware: {
                behaviorClass: FetchStateAwareBehavior
            }
        },

        /**
         * Information about collection fetch state - whether an error occurred, and what content should be displayed
         * in case the error happened.
         */
        isFetchError: false,
        fetchErrorContent: '',

        template: function() {
            return bamboo.plugins.branchinator.renderPullRequestsTableView({
                createPullRequestUrl: this.createPullRequestUrl,
                createPullRequestAnalyticsData: createPullRequestAnalyticsData
            });
        },

        initialize: function() {
            this.fetchCollection();
        },

        fetchCollection: function() {
            clearTimeout(this.fetchTimer);
            this.collection.fetch();
        },

        onCollectionSync: function(collection, response) {
            clearTimeout(this.fetchTimer);
            this.fetchTimer = setTimeout(this.fetchCollection.bind(this), 30000);
            this.createPullRequestUrl = response.createPullRequestUrl;

            if (this.isFetchError) {
                this.isFetchError = false;
            }

            this.render();
        },

        onCollectionError: function(collection, response) {
            clearTimeout(this.fetchTimer);
            this.isFetchError = true;
            this.createPullRequestUrl = void 0;
            this.fetchTimer = setTimeout(this.fetchCollection.bind(this), 30000);
            this.collection.reset();

            var responseJSON = response.responseJSON || {};
            if (responseJSON.authenticationInstanceName) {
                this.fetchErrorContent = bamboo.plugins.branchinator.renderOauthError({
                    instanceName: responseJSON.authenticationInstanceName,
                    redirectUrl: responseJSON.authenticationRedirectUrl
                });

                $(window).off('applinks.auth.completion');
                $(window).on('applinks.auth.completion', function(event, id, success) {
                    if (success) {
                        $(window).off('applinks.auth.completion');

                        // render loading icon instead of info box
                        this.isFetchError = false;
                        this.resetFetchState();
                        this.render();

                        // try to fetch collection again
                        this.fetchCollection();
                    }
                }.bind(this));
            } else if (responseJSON.errorKey === 'pullRequestsNotSupported') {
                this.fetchErrorContent = bamboo.plugins.branchinator.renderPullRequestsNotSupportedInfo();
            } else if (responseJSON.remoteRequestStatus === 401) {
                this.fetchErrorContent = aui.message.warning({
                    content: AJS.I18n.getText('branchinator.pullRequests.error.insufficientPermissions')
                });
            } else {
                this.fetchErrorContent = aui.message.error({ content: bnUtils.getErrorFromResponse(response) });
            }

            this.render();
        },

        onInitialFetchComplete: function() {
            // re-render if the collection is empty, to change loading spinner into 'no builds' message
            if (this.collection.isEmpty()) {
                this.render();
            }
        }
    });
});
