define('branchinator/views/layout/content/builds/builds-table-view', [
    'branchinator/behaviors/fetch-state-aware-behavior',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'branchinator/views/layout/content/builds/builds-table-row-view',
    'marionette'
], function(
    FetchStateAwareBehavior,
    bnUtils,
    radioUtils,
    BuildsTableRowView,
    Marionette
) {
    'use strict';

    var BuildsTableEmptyView = Marionette.ItemView.extend({
        tagName: 'tr',
        className: function() {
            return this.getOption('isFetchError') ? 'fetchError' : 'noResults';
        },

        template: function() {
            return bamboo.plugins.branchinator.renderBuildsTableEmptyView({
                fetched: this.getOption('isFetched'),
                isFetchError: this.getOption('isFetchError'),
                fetchErrorContent: this.getOption('fetchErrorContent')
            });
        }
    });

    /**
     * Displays table of builds for the currently selected branch.
     *
     * Parameters:
     * - collection - builds collection
     *
     * TODO: after Marionette upgrade to 2.4.4+ migrate all 'onChildview...' handlers to 'childEvents' map
     */
    return Marionette.CompositeView.extend({
        id: 'enabledBuilds',

        childView: BuildsTableRowView,
        childViewContainer: 'tbody',

        emptyView: BuildsTableEmptyView,
        emptyViewOptions: function() {
            return {
                isFetchError: this.isFetchError,
                fetchErrorContent: this.fetchErrorContent,
                isFetched: this.isFetched()
            };
        },

        collectionEvents: {
            'sync': 'onCollectionSync',
            'error': 'onCollectionError'
        },

        behaviors: {
            FetchStateAware: {
                behaviorClass: FetchStateAwareBehavior
            }
        },

        /**
         * Information about collection fetch state - whether an error occurred, and what content should be displayed
         * in case the error happened.
         */
        isFetchError: false,
        fetchErrorContent: '',

        template: function() {
            return bamboo.plugins.branchinator.renderBuildsTableView();
        },

        initialize: function() {
            this.fetchCollection();
        },

        fetchCollection: function() {
            clearTimeout(this.fetchTimer);
            this.collection.fetch({ reset: true });
        },

        updateFavicon: function() {
            var favicon;

            if (this.collection.anyBuildFailed()) {
                favicon = '/images/iconsv4/icon-build-failed-64x64.png';
            } else if (this.collection.anyBuildSuccessful()) {
                favicon = '/images/iconsv4/icon-build-successful-64x64.png';
            } else {
                favicon = '/images/icons/favicon.png';
            }

            Piecon.setFavicon(BAMBOO.staticResourcePrefix + favicon)
        },

        onCollectionSync: function() {
            clearTimeout(this.fetchTimer);
            this.fetchTimer = setTimeout(this.fetchCollection.bind(this), 10000);
            this.updateFavicon();

            if (this.isFetchError) {
                this.isFetchError = false;
                this.render();
            }
        },

        onCollectionError: function(collection, response) {
            clearTimeout(this.fetchTimer);
            this.isFetchError = true;
            this.fetchTimer = setTimeout(this.fetchCollection.bind(this), 10000);
            this.collection.reset();

            this.fetchErrorContent = aui.message.error({
                content: bnUtils.getErrorFromResponse(response)
            });

            this.render();
        },

        onInitialFetchComplete: function() {
            // re-render if the collection is empty, to change loading spinner into 'no builds' message
            if (this.collection.isEmpty()) {
                this.render();
            }
        },

        onChildviewBuildStopped: function() {
            this.fetchCollection();
        },

        onChildviewBuildStarted: function() {
            this.fetchCollection();
        },

        onChildviewBuildResumed: function() {
            this.fetchCollection();
        },

        onChildviewBuildDeleted: function(childView) {
            childView.hideWithHighlight().done(this.fetchCollection.bind(this));
        },

        onAddChild: function(childView) {
            // only highlight a row if it's not an initial fetch
            // (call to #isFetched will return false on first rendering, e.g. after user has switched the branch)
            if (this.isFetched() &&
                    childView instanceof BuildsTableRowView &&
                    childView.model.isJustAdded()) {
                childView.showWithHighlight();
            }
        },

        onBeforeDestroy: function() {
            clearTimeout(this.fetchTimer);
        }
    });
});
