define('branchinator/views/layout/content/builds/builds-table-row-view', [
    'branchinator/dialogs/build-commits-dialog',
    'branchinator/dialogs/delete-build-confirmation-dialog',
    'branchinator/utils/branchinator-utils',
    'jquery',
    'marionette'
], function(
    BuildCommitsDialog,
    DeleteBuildConfirmationDialog,
    bnUtils,
    $,
    Marionette
) {
    'use strict';

    /**
     * View for single row of builds table. Handles all actions related to a particular plan branch, e.g. starting,
     * stopping, enabling and deleting.
     *
     * Parameters:
     * - model - a model from builds collection
     */
    return Marionette.ItemView.extend({
        tagName: 'tr',
        className: function() {
            return 'plan-' + this.model.get('planKey');
        },

        events: {
            'click .stopBuild': 'onStopBuildClicked',
            'click .startBuild': 'onStartBuildClicked',
            'click .resumeBuild': 'onResumeBuildClicked',
            'click .editBuild': 'onEditBuildClicked',
            'click .deleteBuild': 'onDeleteBuildClicked',
            'click .commitsLink': 'onCommitsLinkClicked'
        },

        modelEvents: {
            'change': 'render'
        },

        template: function(data) {
            return bamboo.plugins.branchinator.renderBuildsTableRowView({ build: data });
        },

        onStopBuildClicked: function(e) {
            e.preventDefault();
            this.model.stopBuild().done(function() {
                this.trigger('build:stopped', this.model.get('planKey'));
                AJS.trigger('analyticsEvent', { name: 'branch.view.action.build.stopped' });
            }.bind(this));
        },

        onStartBuildClicked: function(e) {
            e.preventDefault();
            this.model.startBuild().done(function() {
                this.trigger('build:started', this.model.get('planKey'));
                AJS.trigger('analyticsEvent', { name: 'branch.view.action.build.started' });
            }.bind(this));
        },

        onResumeBuildClicked: function(e) {
            e.preventDefault();
            this.model.resumeBuild().done(function() {
                this.trigger('build:resumed', this.model.get('planKey'));
                AJS.trigger('analyticsEvent', { name: 'branch.view.action.plan.branch.enabled' });
            }.bind(this));
        },

        onEditBuildClicked: function(e) {
            e.preventDefault();
            AJS.trigger('analyticsEvent', { name: 'branch.view.action.build.configured' });
            window.location.href = $(e.currentTarget).attr('href');
        },

        onDeleteBuildClicked: function(e) {
            e.preventDefault();
            new DeleteBuildConfirmationDialog({ callback: this.deleteBuild.bind(this) });
        },

        onCommitsLinkClicked: function(e) {
            e.preventDefault();
            AJS.trigger('analyticsEvent', { name: 'branch.view.action.commits.show' });
            new BuildCommitsDialog({ planKey: this.model.get('planKey') });
        },

        deleteBuild: function() {
            this.model.deleteBuild().done(function() {
                this.trigger('build:deleted', this.model.get('planKey'));
                AJS.trigger('analyticsEvent', { name: 'branch.view.action.plan.branch.deleted' });
            }.bind(this));
        },

        showWithHighlight: function() {
            return $.Deferred(function(defer) {
                this.$el.addClass('animateHighlight');
                this.$el.one(bnUtils.getAnimationCompletedEventName(), defer.resolve);
            }.bind(this)).promise();
        },

        hideWithHighlight: function() {
            return $.Deferred(function(defer) {
                this.$el.addClass('animateFadeOut');
                this.$el.one(bnUtils.getAnimationCompletedEventName(), function() {
                    // animation only sets opacity to 0, now we hide the entire element so it does not take any space
                    this.$el.hide();
                    defer.resolve();
                }.bind(this));
            }.bind(this)).promise();
        }
    });
});
