define('branchinator/views/layout/content/builds/add-plans-view', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'jquery',
    'marionette'
], function(
    Backbone,
    bnUtils,
    radioUtils,
    $,
    Marionette
) {
    'use strict';

    /**
     * Component for adding new plan branches.
     *
     * Parameters:
     * - model - branch selection model
     */
    return Marionette.ItemView.extend({

        ui: {
            input: '#addPlansSelect',
            addButton: '#addPlansButton'
        },

        events: {
            'click @ui.addButton': 'onAddButtonClicked'
        },

        template: function() {
            return bamboo.plugins.branchinator.renderAddPlansView();
        },

        onRender: function() {
            this.ui.input.auiSelect2({
                placeholder: AJS.I18n.getText('branchinator.available.builds.select.placeholder'),
                minimumInputLength: 0,
                multiple: true,
                ajax: {
                    url: function() {
                        return bnUtils.getRestUrl('builds/available?repoId={0}&branchName={1}', this.model.get('repo').id, encodeURIComponent(this.model.get('branch').branchName));
                    }.bind(this),
                    dataType: 'json',
                    data: function (searchTerm) {
                        return { searchTerm: searchTerm }
                    },
                    results: function(data) {
                        return { results: data };
                    },
                    cache: false
                },
                id: function (result) {
                    return result.planKey;
                },
                formatResult: function(result) {
                    return bamboo.plugins.branchinator.renderBuildName({
                        planName: result.buildName,
                        projectName: result.project.name
                    });
                },
                formatSelection: function(result) {
                    return result.buildName;
                }
            });
        },

        onAddButtonClicked: function(e) {
            e.preventDefault();
            var deferredCalls = [];
            var selectedKeys = this.ui.input.auiSelect2('val');

            // disabled AUI Select2 is not styled properly if it has content, so we clear it beforehand
            this.ui.input.auiSelect2('val', '');

            // disable the input
            this.ui.input.auiSelect2('disable');
            this.ui.addButton.prop('disabled', true);

            // add plans
            selectedKeys.forEach(function(planKey) {
                deferredCalls.push(this.addPlan(planKey));
            }.bind(this));

            $.when.apply($, deferredCalls).fail(function() {
                // if at least one plan was not added successfully
                radioUtils.trigger('message:error', AJS.I18n.getText('branchinator.available.builds.add.error'));
            }).always(function() {
                // always re-enable the input at the end
                this.ui.input.auiSelect2('enable');
                this.ui.addButton.prop('disabled', false);
            }.bind(this));
        },

        addPlan: function(planKey) {
            var branchName = this.model.get('branch').branchName;
            return $.ajax({
                url: bnUtils.getRestUrl('actions/create?planKey={0}&branchName={1}', planKey, encodeURIComponent(branchName)),
                type: 'POST',
                contentType: 'application/json',
                dataType: 'json'
            }).done(function(response) {
                AJS.trigger('analyticsEvent', { name: 'branch.view.action.plan.branch.created' });
                this.triggerMethod('build:added', response.branchKey);
            }.bind(this));
        }
    });
});
