define('branchinator/views/layout/content/builds-view', [
    'branchinator/collections/builds-collection',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'branchinator/views/layout/content/builds/add-plans-view',
    'branchinator/views/layout/content/builds/builds-table-view',
    'jquery',
    'marionette',
    'util/ajax'
], function(
    BuildsCollection,
    bnUtils,
    radioUtils,
    AddPlansView,
    BuildsTableView,
    $,
    Marionette,
    ajax
) {
    'use strict';

    /**
     * Known orderings for builds table.
     * @type {Array<Object>}
     */
    var buildsSortOrderings = WRM.data.claim('com.atlassian.bamboo.plugins.branchinator:branchinator-resources.builds-sort-orderings');

    /**
     * View displayed when user has selected a repository and branch. Will contain builds table and component to add
     * new plan branches.
     *
     * Parameters:
     * - model - branch selection model
     */
    return Marionette.LayoutView.extend({
        regions: {
            addPlans: '#addPlansContainer',
            buildsTable: '#buildsTableContainer'
        },

        ui: {
            sortOrderSelect: '#buildsSortOrderSelect'
        },

        events: {
            'change @ui.sortOrderSelect': 'onSortOrderChanged'
        },

        modelEvents: {
            'repo:cleared repo:selected branch:cleared': 'onBranchCleared',
            'branch:selected': 'onBranchSelected'
        },

        template: function() {
            return bamboo.plugins.branchinator.renderBuildsView({ buildsSortOrderings: buildsSortOrderings });
        },

        showBuildsTable: function() {
            this.buildsTable.show(new BuildsTableView({
                collection: BuildsCollection.forRepositoryBranch(
                    this.model.getRepoId(),
                    this.model.getBranchName(),
                    this.ui.sortOrderSelect.val()
                )
            }));
        },

        saveSortOrderForUser: function() {
            var propertyKey = 'branch.builds.sort.order';
            var data = {
                property: propertyKey,
                value: this.ui.sortOrderSelect.val()
            };

            return ajax({
                url: bnUtils.getRestUrl('config/user/{0}', propertyKey),
                type: 'PUT',
                data: JSON.stringify(data),
                contentType: 'application/json',
                dataType: 'json'
            });
        },

        onRender: function() {
            this.ui.sortOrderSelect.auiSelect2({
                minimumResultsForSearch: Infinity
            });
        },

        onShow: function() {
            this.addPlans.show(new AddPlansView({ model: this.model }));
        },

        onChildviewBuildAdded: function() {
            if (this.buildsTable.currentView) {
                this.buildsTable.currentView.fetchCollection();
            }
        },

        onBranchSelected: function() {
            this.showBuildsTable();
        },

        onBranchCleared: function() {
            this.buildsTable.reset();
        },

        onSortOrderChanged: function() {
            this.saveSortOrderForUser();

            // only refresh the view if builds table is already displayed
            if (this.buildsTable.currentView) {
                this.showBuildsTable();
            }
        }
    });
});
