define('branchinator/models/pull-request-model', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'jquery'
], function(
    Backbone,
    bnUtils,
    $
) {
    'use strict';

    var defaultAvatarUrl = BAMBOO.staticResourcePrefix + '/images/icons/useravatar-grey.png';
    var setDefaultAvatarUrl = function(participant) {
        if (!participant.avatarUrl) {
            participant.avatarUrl = defaultAvatarUrl;
        }
    };

    /**
     * Model representing a single pull request from a specific VCS branch.
     */
    return Backbone.Model.extend({

        /**
         * Attribute under which bamboo repository ID is stored.
         */
        bambooRepoIdAttribute: 'bambooRepoId',

        parse: function(data) {
            if (data.author) {
                setDefaultAvatarUrl(data.author);
            }
            if (data.reviewers) {
                data.reviewers.forEach(setDefaultAvatarUrl);
            }
            return data;
        },

        /**
         * Id of pull request.
         */
        getId: function() {
            return this.get('id');
        },

        /**
         * Id of repository in Bamboo.
         */
        getBambooRepoId: function() {
            return this.get(this.bambooRepoIdAttribute);
        },

        /**
         * URL of this pull request.
         */
        getUrl: function() {
            return this.get('url');
        },

        /**
         * Retrieve properties related to mergeability of the pull request.
         */
        getMergeabilityState: function() {
            return {
                canMerge: this.get('canMerge'),
                conflicted: this.get('conflicted'),
                mergeabilityKnown: this.get('mergeabilityKnown')
            };
        },

        /**
         * Set properties related to mergeability of the pull request. Calling this method will set the flag that
         * mergeability state is known.
         */
        setMergeabilityState: function(canMerge, conflicted) {
            this.set({
                canMerge: canMerge,
                conflicted: conflicted,
                mergeabilityKnown: true
            });
        },

        /**
         * Fetches mergeability state of the pull request from the server.
         */
        fetchMergeabilityState: function() {
            return $.ajax({
                url: bnUtils.getRestUrl('pullRequest/canMerge?repoId={0}&pullRequestId={1}',
                    this.getBambooRepoId(), this.getId()),
                context: this,
                type: 'GET',
                dataType: 'json'
            }).done(function(response) {
                this.setMergeabilityState(response.canMerge, response.conflicted);
            });
        }
    });
});
