define('branchinator/application', [
    'backbone',
    'branchinator/models/branch-selection-model',
    'branchinator/routers/router',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'branchinator/views/layout-view',
    'marionette'
], function(
    Backbone,
    BranchSelectionModel,
    BranchinatorRouter,
    bnUtils,
    radioUtils,
    LayoutView,
    Marionette
) {
    'use strict';

    /**
     * Main branch status page application.
     *
     * Parameters:
     * - el                 - selector for main application container
     * - hasLinkedRepos     - whether there are any linked repositories defined
     * - hasBranchableRepos - whether there are any linked, branchable repositories defined
     */
    return Marionette.Application.extend({

        onStart: function(options) {
            var regionManager = new Marionette.RegionManager({ regions: { main: options.el } });

            this.options = options;
            this.mainRegion = regionManager.get('main');
            this.branchSelection = new BranchSelectionModel();

            this.initializeNavigation();
            this.showPageLayout();
            this.initializeRouting();
            this.displayMessages();
        },

        /**
         * Initializes page navigation. Attaches handlers to branch selection model's events for purpose of browser
         * history manipulation.
         */
        initializeNavigation: function() {
            this.listenTo(this.branchSelection, 'repo:cleared', function() {
                Backbone.history.navigate();
            });
            this.listenTo(this.branchSelection, 'repo:selected branch:cleared', function(repo) {
                Backbone.history.navigate(bnUtils.getRepoUrlPath(repo.id, repo.name));
            });
            this.listenTo(this.branchSelection, 'branch:selected', function(selection) {
                Backbone.history.navigate(bnUtils.getRepoAndBranchUrlPath(selection.repo.id, selection.repo.name, selection.branch.branchName));
            });
        },

        /**
         * Displays the branch status page.
         */
        showPageLayout: function() {
            this.mainRegion.show(new LayoutView({
                model: this.branchSelection,
                hasRepositories: this.options.hasBranchableRepos
            }));
        },

        /**
         * Initializes branch status page routing and starts browser history tracking.
         */
        initializeRouting: function() {
            new BranchinatorRouter({ model: this.branchSelection });
            Backbone.history.start({ pushState: true, root: bnUtils.getApplicationRootUrl() });
        },

        /**
         * Displays initial messages to user if necessary.
         */
        displayMessages: function() {
            if (!this.options.hasLinkedRepos) {
                radioUtils.trigger('message:info', {
                    body: AJS.I18n.getText('branchinator.error.noGlobalRepositories'),
                    close: 'never'
                });
            } else if (!this.options.hasBranchableRepos) {
                radioUtils.trigger('message:info', {
                    body: AJS.I18n.getText('branchinator.error.noBranchAwareRepositories'),
                    close: 'never'
                });
            }
        }
    });
});
