package com.atlassian.aws.s3;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.transfer.Download;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.atlassian.aws.AmazonClients;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class S3Synchroniser
{
    private static final Logger log = Logger.getLogger(S3Synchroniser.class);
    private final AWSCredentials awsCredentials;

    public S3Synchroniser(final AWSCredentials awsCredentials) {
        this.awsCredentials = awsCredentials;
    }

    public void sync(final String srcPath, final String dstPath) throws IOException
    {
        final boolean restrictSyncToSourceDirectories = false;
        sync(srcPath, dstPath, restrictSyncToSourceDirectories);
    }

    private static class FileData
    {
        private final String name;
        protected String md5;
        private final long size;

        private FileData(@NotNull String name, @Nullable String md5, long size)
        {
            this.name = name;
            this.md5 = md5;
            this.size = size;
        }

        public static FileData directory(String name)
        {
            if (!name.endsWith("/"))
            {
                name+="/";
            }
            return new FileData(name, null, -1);
        }

        public static FileData localFile(String name, File file)
        {
            return new LocalFileData(name, file);
        }

        public String getMd5()
        {
            return md5;
        }

        public boolean isTheSame(@Nullable FileData dstFileData)
        {
            if (dstFileData==null ||
                    size!=dstFileData.size)
            {
                log.debug("Different size: " + this + " and " + dstFileData);
                return false;
            }

            final boolean isImmutableFile = !dstFileData.getName().contains("-SNAPSHOT");

            final boolean isTheSame = isImmutableFile || getMd5().equals(dstFileData.getMd5());
            log.debug((isTheSame ? "The same: " : "Different: ") + this + " and " + dstFileData);
            return isTheSame;
        }

        public String getName()
        {
            return name;
        }

        @Override
        public String toString()
        {
            return "FileData{" +
                    "name='" + name + '\'' +
                    ", md5='" + md5 + '\'' +
                    ", size=" + size +
                    '}';
        }
    }

    private static class LocalFileData extends FileData
    {
        private final File file;

        private LocalFileData(String name, File file)
        {
            super(name, null, file.length());
            this.file = file;
        }

        @Override
        public String getMd5()
        {
            if (md5==null)
            {
                try
                {
                    md5 = calculateMd5Hex(file);
                }
                catch (IOException e)
                {
                    throw new RuntimeException(e);
                }
            }
            return md5;
        }
        
        private String calculateMd5Hex(final File assemblyFile) throws IOException {
            try (final FileInputStream fis = new FileInputStream(assemblyFile)) {
                return DigestUtils.md5Hex(fis);
            }
        }
    }

    public void sync(final String srcPath, final String dstPath, final boolean restrictSyncToSourceDirectories) throws IOException
    {
        final File dst = determineOutputDirectory(srcPath, dstPath);
        log.info("Syncing " + srcPath + " to " + dst);

        final S3Path srcS3Path = new S3Path(srcPath);

        final AmazonS3 s3Client = 
                AmazonClients.setBestEndpointForBucket(AmazonClients.newS3Client(awsCredentials), srcS3Path.getBucket())
                        .build();
        final Map<String, FileData> srcObjectNamesAndHashes = getObjectNamesAndHashes(s3Client, srcS3Path);
        final Map<String, FileData> dstObjectNamesAndHashes = getObjectNamesAndHashes(dst);

        final List<String> toGet = listFilesToFetch(srcObjectNamesAndHashes, dstObjectNamesAndHashes);
        final List<String> toRemove = listFilesToRemove(srcObjectNamesAndHashes, dstObjectNamesAndHashes, restrictSyncToSourceDirectories);

        final TransferManager transferManager = EtagCalculator.newCompatibleTransferManager(s3Client);
        try {
            doSync(transferManager, srcS3Path, dst, toGet, toRemove);
        } finally {
            transferManager.shutdownNow();
        }
    }

    private List<String> listFilesToRemove(final Map<String, FileData> srcObjectNamesAndHashes, final Map<String, FileData> dstObjectNamesAndHashes, final boolean restrictRemovalToSourceDirectories)
    {
        log.info("Generating the list of files to remove...");
        final Set<String> dirsBeingSynced = new HashSet<String>();
        if (restrictRemovalToSourceDirectories)
        {
            for (String srcName : srcObjectNamesAndHashes.keySet())
            {
                dirsBeingSynced.add(getTopDirectory(srcName));
            }
        }
        log.debug("Directories being synchronised: " + dirsBeingSynced);
        
        final List<String> toRemove = new LinkedList<String>();
        for (String removalCandidate : dstObjectNamesAndHashes.keySet())
        {
            final boolean fileExistsInSrc = srcObjectNamesAndHashes.containsKey(removalCandidate);
            if (!fileExistsInSrc)
            {
                String removalCandidateDir = getTopDirectory(removalCandidate);

                if (!restrictRemovalToSourceDirectories || dirsBeingSynced.contains(removalCandidateDir))
                {
                    toRemove.add(removalCandidate);
                }
            }
        }
        return toRemove;
    }

    @Nullable
    private String getTopDirectory(@NotNull String path)
    {
        final int slashIndex = path.indexOf("/");
        return (slashIndex == -1) ? "" :  path.substring(0, slashIndex);
    }

    private List<String> listFilesToFetch(final Map<String, FileData> srcObjectNamesAndHashes, final Map<String, FileData> dstObjectNamesAndHashes)
    {
        log.info("Generating the list of files to fetch from S3...");
        final List<String> toGet = new LinkedList<String>();

        for (Map.Entry<String, FileData> srcObject : srcObjectNamesAndHashes.entrySet())
        {
            final String srcName = srcObject.getKey();
            final boolean isDirectory = srcName.endsWith("/");
            if (!isDirectory)
            {
                final FileData srcFileData = srcObject.getValue();
                final FileData dstFileData = dstObjectNamesAndHashes.get(srcName);

                if (!srcFileData.isTheSame(dstFileData))
                {
                    toGet.add(srcName);
                }
            }
        }
        return toGet;
    }

    @NotNull
    private File determineOutputDirectory(final String srcPath, final String dstPath)
    {
        final File dst;
        if (srcPath.endsWith("/"))
        {
            dst = new File(dstPath);
        }
        else
        {
            final String[] pathElements = srcPath.split("/");
            dst = new File(dstPath, pathElements[pathElements.length - 1]);
        }

        return dst;
    }

    private void doSync(final TransferManager transferManager, final S3Path srcPath, final File dstPath, final List<String> toGet, final List<String> toRemove) throws IOException
    {
        log.info("Removing " + toRemove.size() + " files from " + dstPath);
        Collections.sort(toRemove, new Comparator<String>()
        {
            @Override
            public int compare(String o1, String o2)
            {
                return o2.length() - o1.length();
            }
        });
        for (String dstName : toRemove)
        {
            File fileToRemove = new File(dstPath, dstName);
            log.debug("Removing " + fileToRemove);
            if (!fileToRemove.delete())
            {
                throw new FileNotFoundException("Unable to remove file: " + fileToRemove + " - please check write permissions for user " + System.getProperty("user.name"));
            }
        }
        log.info("Fetching " + toGet.size() + " files to " + dstPath);

        final List<Download> downloads = new ArrayList<Download>();
        for (final String srcName : toGet)
        {
            final Download download = fetch(transferManager, srcPath, srcName, dstPath);
            downloads.add(download);
        }
        long totalRead = 0;
        for (final Download download : downloads)
        {
            try
            {
                download.waitForCompletion();
                totalRead += download.getObjectMetadata().getInstanceLength();
            }
            catch (final InterruptedException e)
            {
                throw new RuntimeException(e);
            }
        }

        log.info("Fetched " + totalRead / 1024 / 1024 + " MB from S3");
    }

    private Download fetch(final TransferManager transferManager, final S3Path srcBucket, final String srcName, final File dstPath) throws IOException
    {
        final String srcKey = srcBucket.getKey() + '/' + srcName;
        log.debug(srcBucket.getBucket() + " / " + srcKey);

        final GetObjectRequest getObjectRequest = new GetObjectRequest(srcBucket.getBucket(), srcKey);
        final File outputFile = new File(dstPath, srcName);

        final Download download = transferManager.download(getObjectRequest, outputFile);
        download.addProgressListener(new ProgressListener()
        {
            @Override
            public void progressChanged(final ProgressEvent progressEvent)
            {
                switch (progressEvent.getEventType())
                {
                    case TRANSFER_STARTED_EVENT:
                        log.info("Downloading: " + srcName);
                        break;
                }
            }
        });
        return download;
    }

    private Map<String, FileData> getObjectNamesAndHashes(File dst) throws IOException
    {
        Map<String, FileData> name2file = new HashMap<String, FileData>();

        fillFileDataMap(name2file, dst, dst);

        log.info("Found " + name2file.size() + " files in " + dst);
        return name2file;
    }

    private Map<String, FileData> getObjectNamesAndHashes(final AmazonS3 s3Client, final S3Path s3Location)
    {
        log.info("Fetching the list of remote objects...");
        String s3Prefix = s3Location.getKey();
        ObjectListing objectListing = s3Client.listObjects(s3Location.getBucket(), s3Prefix);

        String s3Directory = getDirectoryFromPrefix(s3Prefix);

        Map<String, FileData> name2file = new HashMap();
        fillFileDataMap(s3Directory, name2file, objectListing);

        while (objectListing.isTruncated())
        {
            objectListing = s3Client.listNextBatchOfObjects(objectListing);
            fillFileDataMap(s3Directory, name2file, objectListing);
        }
        log.info("Found " + name2file.size() + " files in " + s3Location);

        return name2file;
    }

    private String getDirectoryFromPrefix(final String s3Prefix)
    {
        if (s3Prefix.endsWith("/"))
        {
            return s3Prefix.substring(0, s3Prefix.length() - 1);
        }
        return s3Prefix;
    }

    private static void fillFileDataMap(String baseDirectory, Map<String, FileData> name2fileData, ObjectListing objectListing)
    {
        for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries())
        {
            String objectKey = objectSummary.getKey();
            final boolean isDirectory = objectKey.endsWith("/");
            String objectMd5 = isDirectory ? "" : objectSummary.getETag();

            final String fileKey = objectKey.substring(baseDirectory.length() + 1);
            putParentDirs(name2fileData, fileKey);
            name2fileData.put(fileKey, new FileData(objectKey, objectMd5, objectSummary.getSize()));
        }
    }

    private void fillFileDataMap(Map<String, FileData> name2fileData, File baseDir, File file) throws IOException
    {
        int baseDirNameLength = baseDir.getAbsolutePath().length();
        String relativePath;
        if (file.getAbsolutePath().length() > baseDirNameLength)
        {
            relativePath = file.getAbsolutePath().substring(baseDirNameLength + 1);
        }
        else
        {
            relativePath = "";
        }

        if (file.isFile())
        {
            putFsAgnosticData(name2fileData, FileData.localFile(relativePath, file));
        }
        else
        {
            if (!relativePath.isEmpty())
            {
                putFsAgnosticData(name2fileData, FileData.directory(relativePath));
            }
            File[] filesInDir = file.listFiles();
            if (filesInDir == null)
            {
                return;
            }
            for (File fileInDir : filesInDir)
            {
                fillFileDataMap(name2fileData, baseDir, fileInDir);
            }
        }
    }

    private static void putParentDirs(final Map<String, FileData> name2fileData, final String fileKey)
    {
        final String[] pathComponents = fileKey.split("/");
        StringBuilder path = new StringBuilder();
        for (int i = 0; i<pathComponents.length-1; ++i)
        {
            path.append(pathComponents[i]).append("/");
            putFsAgnosticData(name2fileData, FileData.directory(path.toString()));
        }
    }

    private static void putFsAgnosticData(final Map<String, FileData> name2fileData, final FileData fileData)
    {
        String path = fileData.getName();
        name2fileData.put(path.replace(File.separatorChar, '/'), fileData);
    }
}
