package com.atlassian.aws.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.Region;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.Map;

public class S3Support
{
    private S3Support()
    {
    }

    private static final Map<String, Region> S3_REGIONS = new HashMap<String, Region>();

    static
    {
        for (Region region : Region.values())
        {
            S3_REGIONS.put(region.getFirstRegionId(), region);
        }
        S3_REGIONS.remove(Region.US_GovCloud.getFirstRegionId());
        S3_REGIONS.remove(Region.CN_Beijing.getFirstRegionId());
        S3_REGIONS.remove(Region.CN_Northwest_1.getFirstRegionId());
    }

    public static Iterable<Region> getRegions()
    {
        return S3_REGIONS.values();
    }

    public static String getEndpoint(final Region region)
    {
        final com.amazonaws.regions.Region awsRegion = getAwsRegion(region);
        return getEndpoint(awsRegion);
    }

    public static String getEndpoint(final com.amazonaws.regions.Region awsRegion)
    {
        return awsRegion.getServiceEndpoint(AmazonS3.ENDPOINT_PREFIX);
    }

    public static String getName(Region region)
    {
        return getAwsRegion(region).getName();
    }

    @Nullable
    public static String getLocationConstraint(Region region)
    {
        return region.getFirstRegionId();
    }

    private static com.amazonaws.regions.Region getAwsRegion(Region region)
    {
        final com.amazonaws.regions.Region awsRegion = region.toAWSRegion();
        if (awsRegion==null)
        {
            throw new IllegalArgumentException("Unknown AWS region: " + region);
        }
        return awsRegion;
    }


    public static boolean isSameLocation(@Nullable final String locationFromRequest, @NotNull final String locationFromResponse)
    {
        //AWS API is asymmetrical, it's so awesome!
        if (locationFromResponse.equals("US") && locationFromRequest==null)
        {
            return true;
        }
        if (locationFromResponse.equals("EU") && "eu-west-1".equals(locationFromRequest))
        {
            return true;
        }
        return locationFromResponse.equals(locationFromRequest);
    }
}
