/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.aws.s3;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.transfer.Download;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.atlassian.aws.AmazonClients;
import com.atlassian.aws.s3.EtagCalculator;
import com.atlassian.aws.s3.S3Path;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class S3Synchroniser {
    private static final Logger log = Logger.getLogger(S3Synchroniser.class);
    private final AmazonS3 s3Client;

    public S3Synchroniser(AWSCredentials awsCredentials) {
        this.s3Client = AmazonClients.newAmazonS3Client(awsCredentials);
    }

    public void sync(String srcPath, String dstPath) throws IOException {
        boolean restrictSyncToSourceDirectories = false;
        this.sync(srcPath, dstPath, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void sync(String srcPath, String dstPath, boolean restrictSyncToSourceDirectories) throws IOException {
        File dst = this.determineOutputDirectory(srcPath, dstPath);
        log.info((Object)("Syncing " + srcPath + " to " + dst));
        S3Path srcS3Path = new S3Path(srcPath);
        AmazonClients.setBestEndpointForBucket(this.s3Client, srcS3Path.getBucket());
        Map<String, FileData> srcObjectNamesAndHashes = this.getObjectNamesAndHashes(srcS3Path);
        Map<String, FileData> dstObjectNamesAndHashes = this.getObjectNamesAndHashes(dst);
        List<String> toGet = this.listFilesToFetch(srcObjectNamesAndHashes, dstObjectNamesAndHashes);
        List<String> toRemove = this.listFilesToRemove(srcObjectNamesAndHashes, dstObjectNamesAndHashes, restrictSyncToSourceDirectories);
        TransferManager transferManager = EtagCalculator.newCompatibleTransferManager(this.s3Client);
        try {
            this.doSync(transferManager, srcS3Path, dst, toGet, toRemove);
        }
        finally {
            transferManager.shutdownNow();
        }
    }

    private List<String> listFilesToRemove(Map<String, FileData> srcObjectNamesAndHashes, Map<String, FileData> dstObjectNamesAndHashes, boolean restrictRemovalToSourceDirectories) {
        log.info((Object)"Generating the list of files to remove...");
        HashSet<String> dirsBeingSynced = new HashSet<String>();
        if (restrictRemovalToSourceDirectories) {
            for (String srcName : srcObjectNamesAndHashes.keySet()) {
                dirsBeingSynced.add(this.getTopDirectory(srcName));
            }
        }
        log.debug((Object)("Directories being synchronised: " + dirsBeingSynced));
        LinkedList<String> toRemove = new LinkedList<String>();
        for (String removalCandidate : dstObjectNamesAndHashes.keySet()) {
            boolean fileExistsInSrc = srcObjectNamesAndHashes.containsKey(removalCandidate);
            if (fileExistsInSrc) continue;
            String removalCandidateDir = this.getTopDirectory(removalCandidate);
            if (restrictRemovalToSourceDirectories && !dirsBeingSynced.contains(removalCandidateDir)) continue;
            toRemove.add(removalCandidate);
        }
        return toRemove;
    }

    @Nullable
    private String getTopDirectory(@NotNull String path) {
        int slashIndex = path.indexOf("/");
        return slashIndex == -1 ? "" : path.substring(0, slashIndex);
    }

    private List<String> listFilesToFetch(Map<String, FileData> srcObjectNamesAndHashes, Map<String, FileData> dstObjectNamesAndHashes) {
        log.info((Object)"Generating the list of files to fetch from S3...");
        LinkedList<String> toGet = new LinkedList<String>();
        for (Map.Entry<String, FileData> srcObject : srcObjectNamesAndHashes.entrySet()) {
            FileData dstFileData;
            FileData srcFileData;
            String srcName = srcObject.getKey();
            boolean isDirectory = srcName.endsWith("/");
            if (isDirectory || (srcFileData = srcObject.getValue()).isTheSame(dstFileData = dstObjectNamesAndHashes.get(srcName))) continue;
            toGet.add(srcName);
        }
        return toGet;
    }

    @NotNull
    private File determineOutputDirectory(String srcPath, String dstPath) {
        File dst;
        if (srcPath.endsWith("/")) {
            dst = new File(dstPath);
        } else {
            String[] pathElements = srcPath.split("/");
            dst = new File(dstPath, pathElements[pathElements.length - 1]);
        }
        return dst;
    }

    private void doSync(TransferManager transferManager, S3Path srcPath, File dstPath, List<String> toGet, List<String> toRemove) throws IOException {
        log.info((Object)("Removing " + toRemove.size() + " files from " + dstPath));
        Collections.sort(toRemove, new Comparator<String>(){

            @Override
            public int compare(String o1, String o2) {
                return o2.length() - o1.length();
            }
        });
        for (String string : toRemove) {
            File fileToRemove = new File(dstPath, string);
            log.debug((Object)("Removing " + fileToRemove));
            if (fileToRemove.delete()) continue;
            throw new FileNotFoundException("Unable to remove file: " + fileToRemove + " - please check write permissions for user " + System.getProperty("user.name"));
        }
        log.info((Object)("Fetching " + toGet.size() + " files to " + dstPath));
        ArrayList<Download> downloads = new ArrayList<Download>();
        for (String srcName : toGet) {
            Download download = this.fetch(transferManager, srcPath, srcName, dstPath);
            downloads.add(download);
        }
        long l = 0L;
        for (Download download : downloads) {
            try {
                download.waitForCompletion();
                l += download.getObjectMetadata().getInstanceLength();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        log.info((Object)("Fetched " + l / 1024L / 1024L + " MB from S3"));
    }

    private Download fetch(TransferManager transferManager, S3Path srcBucket, final String srcName, File dstPath) throws IOException {
        String srcKey = srcBucket.getKey() + '/' + srcName;
        log.debug((Object)(srcBucket.getBucket() + " / " + srcBucket.getKey() + "/" + srcName));
        GetObjectRequest getObjectRequest = new GetObjectRequest(srcBucket.getBucket(), srcKey);
        File outputFile = new File(dstPath, srcName);
        Download download = transferManager.download(getObjectRequest, outputFile);
        download.addProgressListener(new ProgressListener(){

            public void progressChanged(ProgressEvent progressEvent) {
                switch (progressEvent.getEventType()) {
                    case TRANSFER_STARTED_EVENT: {
                        log.info((Object)("Downloading: " + srcName));
                    }
                }
            }
        });
        return download;
    }

    private Map<String, FileData> getObjectNamesAndHashes(File dst) throws IOException {
        HashMap<String, FileData> name2file = new HashMap<String, FileData>();
        this.fillFileDataMap(name2file, dst, dst);
        log.info((Object)("Found " + name2file.size() + " files in " + dst));
        return name2file;
    }

    private Map<String, FileData> getObjectNamesAndHashes(S3Path s3Location) {
        log.info((Object)"Fetching the list of remote objects...");
        String s3Prefix = s3Location.getKey();
        ObjectListing objectListing = this.s3Client.listObjects(s3Location.getBucket(), s3Prefix);
        String s3Directory = this.getDirectoryFromPrefix(s3Prefix);
        HashMap<String, FileData> name2file = new HashMap<String, FileData>();
        S3Synchroniser.fillFileDataMap(s3Directory, name2file, objectListing);
        while (objectListing.isTruncated()) {
            objectListing = this.s3Client.listNextBatchOfObjects(objectListing);
            S3Synchroniser.fillFileDataMap(s3Directory, name2file, objectListing);
        }
        log.info((Object)("Found " + name2file.size() + " files in " + s3Location));
        return name2file;
    }

    private String getDirectoryFromPrefix(String s3Prefix) {
        if (s3Prefix.endsWith("/")) {
            return s3Prefix.substring(0, s3Prefix.length() - 1);
        }
        return s3Prefix;
    }

    private static void fillFileDataMap(String baseDirectory, Map<String, FileData> name2fileData, ObjectListing objectListing) {
        for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries()) {
            String objectKey = objectSummary.getKey();
            boolean isDirectory = objectKey.endsWith("/");
            String objectMd5 = isDirectory ? "" : objectSummary.getETag();
            String fileKey = objectKey.substring(baseDirectory.length() + 1);
            S3Synchroniser.putParentDirs(name2fileData, fileKey);
            name2fileData.put(fileKey, new FileData(objectKey, objectMd5, objectSummary.getSize()));
        }
    }

    private void fillFileDataMap(Map<String, FileData> name2fileData, File baseDir, File file) throws IOException {
        int baseDirNameLength = baseDir.getAbsolutePath().length();
        String relativePath = file.getAbsolutePath().length() > baseDirNameLength ? file.getAbsolutePath().substring(baseDirNameLength + 1) : "";
        if (file.isFile()) {
            S3Synchroniser.putFsAgnosticData(name2fileData, FileData.localFile(relativePath, file));
        } else {
            File[] filesInDir;
            if (!relativePath.isEmpty()) {
                S3Synchroniser.putFsAgnosticData(name2fileData, FileData.directory(relativePath));
            }
            if ((filesInDir = file.listFiles()) == null) {
                return;
            }
            for (File fileInDir : filesInDir) {
                this.fillFileDataMap(name2fileData, baseDir, fileInDir);
            }
        }
    }

    private static void putParentDirs(Map<String, FileData> name2fileData, String fileKey) {
        String[] pathComponents = fileKey.split("/");
        StringBuilder path = new StringBuilder();
        for (int i = 0; i < pathComponents.length - 1; ++i) {
            path.append(pathComponents[i]).append("/");
            S3Synchroniser.putFsAgnosticData(name2fileData, FileData.directory(path.toString()));
        }
    }

    private static void putFsAgnosticData(Map<String, FileData> name2fileData, FileData fileData) {
        String path = fileData.getName();
        name2fileData.put(path.replace(File.separatorChar, '/'), fileData);
    }

    private static class LocalFileData
    extends FileData {
        private final File file;

        private LocalFileData(String name, File file) {
            super(name, null, file.length());
            this.file = file;
        }

        @Override
        public String getMd5() {
            if (this.md5 == null) {
                try {
                    this.md5 = this.calculateMd5Hex(this.file);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            return this.md5;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private String calculateMd5Hex(File assemblyFile) throws IOException {
            try (FileInputStream fis = new FileInputStream(assemblyFile);){
                String string = DigestUtils.md5Hex((InputStream)fis);
                return string;
            }
        }
    }

    private static class FileData {
        private final String name;
        protected String md5;
        private final long size;

        private FileData(@NotNull String name, @Nullable String md5, long size) {
            this.name = name;
            this.md5 = md5;
            this.size = size;
        }

        public static FileData directory(String name) {
            if (!name.endsWith("/")) {
                name = name + "/";
            }
            return new FileData(name, null, -1L);
        }

        public static FileData localFile(String name, File file) {
            return new LocalFileData(name, file);
        }

        public String getMd5() {
            return this.md5;
        }

        public boolean isTheSame(@Nullable FileData dstFileData) {
            if (dstFileData == null || this.size != dstFileData.size) {
                log.debug((Object)("Different size: " + this + " and " + dstFileData));
                return false;
            }
            boolean isImmutableFile = !dstFileData.getName().contains("-SNAPSHOT");
            boolean isTheSame = isImmutableFile || this.getMd5().equals(dstFileData.getMd5());
            log.debug((Object)((isTheSame ? "The same: " : "Different: ") + this + " and " + dstFileData));
            return isTheSame;
        }

        public String getName() {
            return this.name;
        }

        public String toString() {
            return "FileData{name='" + this.name + '\'' + ", md5='" + this.md5 + '\'' + ", size=" + this.size + '}';
        }
    }
}

