package com.atlassian.aws.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.internal.ServiceUtils;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.io.File;

/**
 * The method
 * {@link com.amazonaws.services.s3.internal.ServiceUtils#retryableDownloadS3ObjectToFile(File, ServiceUtils.RetryableS3DownloadTask, boolean)},
 * used by
 * {@link com.amazonaws.services.s3.AmazonS3Client#getObject(com.amazonaws.services.s3.model.GetObjectRequest, java.io.File)} does not retry on SocketExceptions and SSLProtocol exception.
 * Which sucks. This class subclasses{@link com.amazonaws.services.s3.AmazonS3Client} and provides an additional method
 * {@link BambooAmazonS3Client#getObjectWithRetries(com.amazonaws.services.s3.model.GetObjectRequest, java.io.File)}
 * that retries the download according to the retry
 * definition from {@link org.apache.http.client.HttpClient} configuration.
 *
 * @deprecated since 5.10 use {@link com.amazonaws.services.s3.transfer.TransferManager} instead
 */
@Deprecated
public class BambooAmazonS3Client extends AmazonS3Client
{
    private static final Logger log = Logger.getLogger(BambooAmazonS3Client.class);

    public BambooAmazonS3Client(final AWSCredentials awsCredentials, final ClientConfiguration clientConfiguration)
    {
        super(awsCredentials, clientConfiguration);
    }

    @Nullable
    public ObjectMetadata getObjectWithRetries(final GetObjectRequest getObjectRequest, final File destinationFile) throws AmazonClientException, AmazonServiceException
    {
        if (destinationFile==null)
        {
            throw new NullPointerException("The destination file parameter must be specified when downloading an object directly to a file");
        }

        final S3Object s3Object = retryableDownloadS3ObjectToFile(destinationFile,
                new ServiceUtils.RetryableS3DownloadTask()
                {
                    @Override
                    public S3Object getS3ObjectStream()
                    {
                        return getObject(getObjectRequest);
                    }

                    @Override
                    public boolean needIntegrityCheck()
                    {
                        return getObjectRequest.getRange() == null;
                    }
                });
        return s3Object!=null ? s3Object.getObjectMetadata() : null;
    }

    @Nullable
    private S3Object retryableDownloadS3ObjectToFile(final File file, final ServiceUtils.RetryableS3DownloadTask retryableS3DownloadTask)
    {
        int retries = clientConfiguration.getMaxErrorRetry();

        S3Object s3Object;
        do
        {
            s3Object = retryableS3DownloadTask.getS3ObjectStream();
            if (s3Object==null)
            {
                return null;
            }
            try
            {
                ServiceUtils.downloadObjectToFile(s3Object, file, retryableS3DownloadTask.needIntegrityCheck(), false);
                return s3Object;
            }
            catch (final AmazonClientException ace)
            {
                if (retries--==0)
                {
                    throw ace;
                }
                log.info("Retry the download of object " + s3Object.getKey() + " (bucket " + s3Object.getBucketName() + ")", ace);
            }
            finally
            {
                abortQuietly(s3Object);
            }
        } while (true);
    }

    private static void abortQuietly(final S3Object s3Object)
    {
        s3Object.getObjectContent().abort();
    }
}