package com.atlassian.application.api;

/**
 * A {@link Application} provided by a set of plugins. There are three types of plugins:
 *
 * <ul>
 *     <li>
 *         Primary Plugin: The plugin that defines the application. Disabling this plugin will will disable the
 *         application. There is only one primary plugin per application. The primary plugin may also provide
 *         application functionality.
 *      </li>
 *      <li>
 *          Application Plugin: A plugin providing fundamental features for the application. All application plugins
 *          must be enabled and working for the application to be considered working. The application plugin is
 *          <strong>owned</strong> by the application (aka the plugin will not work without the primary of the
 *          application).
 *      </li>
 *      <li>Utility Plugin: A plugin that provides utilities to the primary and/or application plugins. A utility plugin
 *          is not owned by the application (i.e. the plugin will work without the primary of the application) and may
 *          be used by multiple applications or other plugins
 *      </li>
 * </ul>
 *
 * Unless otherwise noted none of the parameters or return values can be {@code null}.
 *
 * @since v1.0
 */
public interface PluginApplication extends Application
{
    /**
     * Return the module key for the module in the Primary plugin that defines the application. The plugin key must
     * be included.
     *
     * @return the module key to the module in the Primary plugin that defines the application.
     */
    String getDefinitionModuleKey();

    /**
     * Get the primary, application and utility plugins associated with the application.
     *
     * @return the primary, application and utility plugins associated with the application.
     */
    Iterable<ApplicationPlugin> getPlugins();

    /**
     * Get the primary plugin associated in the application.
     *
     * @return the primary plugin for the application.
     */
    ApplicationPlugin getPrimaryPlugin();

    /**
     * Return all the application plugins in the application.
     *
     * @return the application plugins in the application. Can be empty if the primary provides all the functionality
     * of the application.
     */
    Iterable<ApplicationPlugin> getApplicationPlugins();

    /**
     * Return all the utility plugins in the application.
     *
     * @return the utility plugins in the application. Can be empty.
     */
    Iterable<ApplicationPlugin> getUtilityPlugins();
}
