package com.atlassian.analytics.client.configuration;

import com.atlassian.analytics.api.services.AnalyticsConfigService;
import com.atlassian.analytics.client.detect.OnDemandDetector;
import com.atlassian.cache.*;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.text.SimpleDateFormat;
import java.util.Date;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

public class AnalyticsConfig implements AnalyticsConfigService
{
    static final String KEY_PREFIX = "com.atlassian.analytics.client.configuration.";
    private static final String CACHE_KEY = KEY_PREFIX + "settings-cache";
    private static final Logger LOG = LoggerFactory.getLogger(AnalyticsConfig.class);

    private final PluginSettingsFactory pluginSettingsFactory;
    private final EventPublisher eventPublisher;
    private final OnDemandDetector onDemandDetector;
    private final Cache<String, Boolean> settingsCache;

    public AnalyticsConfig(PluginSettingsFactory pluginSettingsFactory, EventPublisher eventPublisher,
                           OnDemandDetector onDemandDetector, CacheManager cacheManager)
    {
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.eventPublisher = eventPublisher;
        this.onDemandDetector = onDemandDetector;
        this.settingsCache = getSettingsCache(cacheManager);
    }

    private Cache<String, Boolean> getSettingsCache(CacheManager cacheManager)
    {
        // Use a <String, Boolean> Cache instead of a <Key, Boolean> cache to prevent classloader issues (at least in
        // Bitbucket Server)
        CacheLoader<String, Boolean> cacheLoader = new CacheLoader<String, Boolean>()
        {
            @Override
            public Boolean load(@Nonnull String key)
            {
                return getBooleanSetting(Key.valueOf(key));
            }
        };

        CacheSettings cacheSettings = new CacheSettingsBuilder().
                remote().
                replicateAsynchronously().
                replicateViaInvalidation().
                build();

        return cacheManager.getCache(CACHE_KEY, cacheLoader, cacheSettings);
    }

    public String getDestination()
    {
        return getSetting(Key.DESTINATION);
    }

    public String getDestinationOrDefault(String defaultDestination)
    {
        String destination = getDestination();
        return isNotEmpty(destination) ? destination : defaultDestination;
    }

    public void setDestination(String destination)
    {
        checkNotNull(destination);
        updateSetting(Key.DESTINATION, destination);
    }

    public boolean isPolicyUpdateAcknowledged()
    {
        return settingsCache.get(Key.POLICY_ACKNOWLEDGED.name());
    }

    public void setPolicyUpdateAcknowledged(boolean acknowledged)
    {
        boolean updateSuccessful = updateSetting(Key.POLICY_ACKNOWLEDGED, acknowledged);
        if (updateSuccessful)
        {
            settingsCache.put(Key.POLICY_ACKNOWLEDGED.name(), acknowledged);
        }
    }

    public void setDefaultAnalyticsEnabled()
    {
        final String analyticsEnabled = getSetting(Key.ANALYTICS_ENABLED);
        if (isEmpty(analyticsEnabled))
        {
            setAnalyticsEnabled(true, "");
        }
    }

    @Override
    public boolean isAnalyticsEnabled()
    {
        return settingsCache.get(Key.ANALYTICS_ENABLED.name());
    }

    public void setAnalyticsEnabled(boolean analyticsEnabled, final String userName)
    {
        boolean updateSuccessful = updateSetting(Key.ANALYTICS_ENABLED, analyticsEnabled);
        if (updateSuccessful)
        {
            settingsCache.put(Key.ANALYTICS_ENABLED.name(), analyticsEnabled);
        }

        // If someone has disabled analytics, the only way we can trace this is to store something in JIRA
        if (!analyticsEnabled)
        {
            String currentDateTime = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date());
            updateSetting(Key.ANALYTICS_DISABLED_USERNAME, userName, false);
            updateSetting(Key.ANALYTICS_DISABLED_DATE, currentDateTime, false);
        }
    }

    private boolean updateSetting(Key key, boolean newValue)
    {
        return updateSetting(key, Boolean.toString(newValue), true);
    }

    private boolean updateSetting(Key key, String newValue)
    {
        return updateSetting(key, newValue, true);
    }

    private boolean updateSetting(Key key, String newValue, boolean fireEvent)
    {
        String oldValue = getSetting(key);
        boolean putSuccessful = putSetting(key, newValue);
        if (fireEvent && putSuccessful)
        {
            eventPublisher.publish(new AnalyticsConfigChangedEvent(key, oldValue, newValue));
        }
        return putSuccessful;
    }

    private boolean putSetting(final Key key, final String newValue)
    {
        try
        {
            pluginSettingsFactory.createGlobalSettings().put(key.getKey(), newValue);
            return true;
        }
        catch (RuntimeException e)
        {
            LOG.warn("Couldn't change the analytics settings. This can safely be ignored during plugin shutdown. Detail: " + e.getMessage());
        }
        return false;
    }

    private String getSetting(Key key)
    {
        try
        {
            final String setting = (String) pluginSettingsFactory.createGlobalSettings().get(key.getKey());
            return setting == null ? "" : setting;
        }
        catch (RuntimeException e)
        {
            LOG.warn("Couldn't check the analytics settings. This can safely be ignored during plugin shutdown. Detail: " + e.getMessage());
        }
        return "";
    }

    private Boolean getBooleanSetting(Key key)
    {
        String value = getSetting(key);
        return isNotEmpty(value) && Boolean.parseBoolean(value);
    }

    @Override
    public boolean canCollectAnalytics()
    {
        return onDemandDetector.isOnDemand() || (isAnalyticsEnabled() && isPolicyUpdateAcknowledged());
    }

    public boolean hasLoggedBaseData()
    {
        return getBooleanSetting(Key.LOGGED_BASE_DATA);
    }

    public void setLoggedBaseData(final boolean loggedBaseData)
    {
        updateSetting(Key.LOGGED_BASE_DATA, loggedBaseData);
    }

    public enum Key
    {
        DESTINATION("destination"),
        POLICY_ACKNOWLEDGED("policy_acknowledged"),
        ANALYTICS_ENABLED("analytics_enabled"),
        ANALYTICS_DISABLED_USERNAME("analytics_disabled_username"),
        ANALYTICS_DISABLED_DATE("analytics_disabled_date"),
        LOGGED_BASE_DATA("logged_base_analytics_data");

        private final String key;

        Key(String suffix)
        {
            this.key = KEY_PREFIX + "." + suffix;
        }

        public String getKey()
        {
            return key;
        }
    }
}
