package com.apandroid.chiptextview

import android.annotation.TargetApi
import android.content.Context
import android.content.res.TypedArray
import android.graphics.*
import android.graphics.drawable.Drawable
import android.os.Build
import android.text.TextPaint
import android.util.AttributeSet
import android.view.View
import kotlin.math.max
import kotlin.math.roundToInt

open class ChipTextView(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : View(context, attrs, defStyleAttr) {

    private val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
    private val backgroundPaint = Paint(Paint.ANTI_ALIAS_FLAG)
    private val textMeasuringRect = Rect()
    private val chipBackgroundRect = RectF()
    private val chipBackgroundDrawableRect = Rect()

    var chipBackgroundDrawable: RoundedCornersDrawable? = null
        private set

    var text: String? = null
        set(value) {
            field = value
            requestLayout()
        }

    var textSize
        get() = textPaint.textSize
        set(value) {
            textPaint.textSize = value
            requestLayout()
        }

    var textColor
        get() = textPaint.color
        set(value) {
            textPaint.color = value
            invalidate()
        }

    var font
        get() = textPaint.typeface
        set(value) {
            textPaint.typeface = value
            requestLayout()
        }

    var paintFlags
        get() = textPaint.flags
        set(value) {
            textPaint.flags = value
            requestLayout()
        }

    var chipPaddingLeft = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingRight = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingTop = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingBottom = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipBackgroundColor: Int
        get() = backgroundPaint.color
        set(value) {
            backgroundPaint.color = value
            invalidate()
        }

    init {
        attrs?.let { readAttributes(context, it, defStyleAttr) }
    }

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, R.attr.chipTextViewStyle)

    constructor(context: Context) : this(context, null)

    private fun readAttributes(context: Context, attrs: AttributeSet, defStyleAttr: Int) {
        val array = context.obtainStyledAttributes(attrs, R.styleable.ChipTextView, defStyleAttr, R.style.ChipTextViewDefaultStyle)

        chipBackgroundColor = array.getColor(R.styleable.ChipTextView_ctv_chipColor, Color.BLACK)
        setChipBackgroundDrawable(array.getDrawable(R.styleable.ChipTextView_ctv_chipBackgroundDrawable))
        readTextAttributes(array)
        readFont(array)
        readChipPadding(array)
        array.recycle()
    }

    private fun readTextAttributes(array: TypedArray) {
        text = array.getString(R.styleable.ChipTextView_ctv_text)

        textPaint.apply {
            textSize = array.getDimension(R.styleable.ChipTextView_ctv_textSize, 0f)
            color = array.getColor(R.styleable.ChipTextView_ctv_textColor, Color.WHITE)
        }
    }

    private fun readFont(array: TypedArray) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            font = array.getFont(R.styleable.ChipTextView_ctv_font)
        }
    }

    private fun readChipPadding(array: TypedArray) {
        array.apply {
            val padding = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPadding, 0)
            val paddingHorizontal = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingHorizontal, padding)
            val paddingVertical = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingVertical, padding)

            setChipPadding(padding)
            setChipHorizontalPadding(paddingHorizontal)
            setChipVerticalPadding(paddingVertical)

            chipPaddingLeft = readAliasDimension(array, R.styleable.ChipTextView_ctv_chipPaddingLeft, R.styleable.ChipTextView_ctv_chipPaddingStart, paddingHorizontal)
            chipPaddingRight = readAliasDimension(array, R.styleable.ChipTextView_ctv_chipPaddingRight, R.styleable.ChipTextView_ctv_chipPaddingEnd, paddingHorizontal)
            chipPaddingTop = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingTop, paddingVertical)
            chipPaddingBottom = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingBottom, paddingVertical)
        }
    }

    private fun readAliasDimension(array: TypedArray, originName: Int, alias: Int, defaultValue: Int) = array.run {
        if (hasValue(alias)) {
            getDimensionPixelSize(alias, 0)
        } else {
            getDimensionPixelSize(originName, defaultValue)
        }
    }

    fun setChipPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipHorizontalPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
    }

    fun setChipVerticalPadding(padding: Int) {
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipPadding(left: Int, top: Int, right: Int, bottom: Int) {
        chipPaddingLeft = left
        chipPaddingTop = top
        chipPaddingRight = right
        chipPaddingBottom = bottom
    }

    fun setChipBackgroundDrawable(drawable: Drawable?) {
        chipBackgroundDrawable = drawable?.let {
            RoundedCornersDrawable(drawable).also { drawable ->
                drawable.callback = this
                drawable.state = drawableState
                invalidate()
            }
        }
    }

    fun strikeThroughText() {
        paintFlags = paintFlags or TextPaint.STRIKE_THRU_TEXT_FLAG
        invalidate()
    }

    fun hideStrikeThroughText() {
        paintFlags = paintFlags and TextPaint.STRIKE_THRU_TEXT_FLAG.inv()
        invalidate()
    }

    fun underlineText() {
        paintFlags = paintFlags or TextPaint.UNDERLINE_TEXT_FLAG
        invalidate()
    }

    fun hideTextUnderline() {
        paintFlags = paintFlags and TextPaint.UNDERLINE_TEXT_FLAG.inv()
        invalidate()
    }

    override fun verifyDrawable(who: Drawable?) = who
            ?.let { super.verifyDrawable(it) || it === chipBackgroundDrawable } ?: false

    override fun drawableStateChanged() {
        super.drawableStateChanged()
        chipBackgroundDrawable?.state = drawableState
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun drawableHotspotChanged(x: Float, y: Float) {
        super.drawableHotspotChanged(x, y)
        chipBackgroundDrawable?.setHotspot(x, y)
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)

        var preferredWidth = 0
        var preferredHeight = 0

        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it).roundToInt()
            val textHeight = textMeasuringRect.height()

            if (it.length == 1) {
                val max = max(textWidth, textHeight)
                preferredWidth = max
                preferredHeight = max
            } else {
                preferredWidth = textWidth
                preferredHeight = textHeight
            }

            preferredWidth += paddingLeft + paddingRight + chipPaddingLeft + chipPaddingRight
            preferredHeight += paddingTop + paddingBottom + chipPaddingTop + chipPaddingBottom
        }

        val finalWidth = View.resolveSize(preferredWidth, widthMeasureSpec)
        val finalHeight = View.resolveSize(preferredHeight, heightMeasureSpec)

        setMeasuredDimension(finalWidth, finalHeight)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        calcChipBackgroundRect()
        drawChipBackgroundColor(canvas)
        drawChipBackgroundDrawable(canvas)
        drawText(canvas)
    }

    private fun calcChipBackgroundRect() {
        chipBackgroundRect.apply {
            left = paddingLeft.toFloat()
            top = paddingTop.toFloat()
            right = (width - paddingRight).toFloat()
            bottom = (height - paddingBottom).toFloat()
            round(chipBackgroundDrawableRect)
        }
    }

    private fun drawChipBackgroundColor(canvas: Canvas) {
        val cornerRadius = chipBackgroundRect.height() / 2
        canvas.drawRoundRect(chipBackgroundRect, cornerRadius, cornerRadius, backgroundPaint)
    }

    private fun drawChipBackgroundDrawable(canvas: Canvas) {
        chipBackgroundDrawable?.apply {
            bounds = chipBackgroundDrawableRect
            draw(canvas)
        }
    }

    private fun drawText(canvas: Canvas) {
        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it)
            val textHeight = textMeasuringRect.height()
            val clientWidth = width - paddingLeft - paddingRight - chipPaddingLeft - chipPaddingRight
            val clientHeight = height - paddingTop - paddingBottom - chipPaddingTop - chipPaddingBottom
            val textX = paddingLeft + chipPaddingLeft + (clientWidth - textWidth) / 2f
            val textY = paddingTop + chipPaddingTop - textMeasuringRect.top + (clientHeight - textHeight) / 2f

            canvas.drawText(it, textX, textY, textPaint)
        }
    }
}