package com.apandroid.chiptextview

import android.content.Context
import android.content.res.TypedArray
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.os.Build
import android.text.TextPaint
import android.util.AttributeSet
import android.view.View
import kotlin.math.max
import kotlin.math.roundToInt

open class ChipTextView(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : View(context, attrs, defStyleAttr) {

    private val textMeasuringRect = Rect()
    private val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
    private var chipBackgroundDrawable = ChipBackgroundDrawable()

    var text: String? = null
        set(value) {
            field = value
            requestLayout()
        }

    var textSize
        get() = textPaint.textSize
        set(value) {
            textPaint.textSize = value
            requestLayout()
        }

    var textColor
        get() = textPaint.color
        set(value) {
            textPaint.color = value
            invalidate()
        }

    var font
        get() = textPaint.typeface
        set(value) {
            textPaint.typeface = value
            requestLayout()
        }

    var paintFlags
        get() = textPaint.flags
        set(value) {
            textPaint.flags = value
            requestLayout()
        }

    var chipPaddingLeft = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingRight = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingTop = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingBottom = 0
        set(value) {
            field = value
            requestLayout()
        }

    val chipDrawable get() = chipBackgroundDrawable

    init {
        attrs?.let { parseAttributes(context, it, defStyleAttr) }
    }

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, R.attr.chipTextViewStyle)

    constructor(context: Context) : this(context, null)

    private fun parseAttributes(context: Context, attrs: AttributeSet, defStyleAttr: Int) {
        val array = context.obtainStyledAttributes(attrs, R.styleable.ChipTextView, defStyleAttr, R.style.ChipTextViewDefaultStyle)

        text = array.getString(R.styleable.ChipTextView_ctv_text)
        textPaint.textSize = array.getDimension(R.styleable.ChipTextView_ctv_textSize, 0f)
        textPaint.color = array.getColor(R.styleable.ChipTextView_ctv_textColor, Color.WHITE)

        setChipColor(array.getColor(R.styleable.ChipTextView_ctv_chipColor, Color.BLACK))
        readFont(array)
        readChipPadding(array)
        array.recycle()
    }

    private fun readFont(array: TypedArray) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            font = array.getFont(R.styleable.ChipTextView_ctv_font)
        }
    }

    private fun readChipPadding(array: TypedArray) {
        array.apply {
            val padding = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPadding, 0)
            val paddingHorizontal = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingHorizontal, padding)
            val paddingVertical = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingVertical, padding)

            setChipPadding(padding)
            setChipHorizontalPadding(paddingHorizontal)
            setChipVerticalPadding(paddingVertical)

            chipPaddingLeft = readAliasDimension(array, R.styleable.ChipTextView_ctv_chipPaddingLeft, R.styleable.ChipTextView_ctv_chipPaddingStart, paddingHorizontal)
            chipPaddingRight = readAliasDimension(array, R.styleable.ChipTextView_ctv_chipPaddingRight, R.styleable.ChipTextView_ctv_chipPaddingEnd, paddingHorizontal)
            chipPaddingTop = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingTop, paddingVertical)
            chipPaddingBottom = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingBottom, paddingVertical)
        }
    }

    private fun readAliasDimension(array: TypedArray, originName: Int, alias: Int, defaultValue: Int) = array.run {
        if (hasValue(alias)) {
            getDimensionPixelSize(alias, 0)
        } else {
            getDimensionPixelSize(originName, defaultValue)
        }
    }

    fun setChipPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipHorizontalPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
    }

    fun setChipVerticalPadding(padding: Int) {
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipPadding(left: Int, top: Int, right: Int, bottom: Int) {
        chipPaddingLeft = left
        chipPaddingTop = top
        chipPaddingRight = right
        chipPaddingBottom = bottom
    }

    fun setChipColor(color: Int) {
        chipBackgroundDrawable = ChipBackgroundDrawable(color)
        invalidate()
    }

    fun strikeThroughText() {
        paintFlags = paintFlags or TextPaint.STRIKE_THRU_TEXT_FLAG
    }

    fun hideStrikeThroughText() {
        paintFlags = paintFlags and TextPaint.STRIKE_THRU_TEXT_FLAG.inv()
    }

    fun underlineText() {
        paintFlags = paintFlags or TextPaint.UNDERLINE_TEXT_FLAG
    }

    fun hideTextUnderline() {
        paintFlags = paintFlags and TextPaint.UNDERLINE_TEXT_FLAG.inv()
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)

        var preferredWidth= 0
        var preferredHeight = 0

        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it).roundToInt()
            val textHeight = textMeasuringRect.height()

            if (it.length == 1) {
                val max = max(textWidth, textHeight)
                preferredWidth = max
                preferredHeight = max
            } else {
                preferredWidth = textWidth
                preferredHeight = textHeight
            }

            preferredWidth += paddingLeft + paddingRight + chipPaddingLeft + chipPaddingRight
            preferredHeight += paddingTop + paddingBottom + chipPaddingTop + chipPaddingBottom
        }

        val finalWidth = View.resolveSize(preferredWidth, widthMeasureSpec)
        val finalHeight = View.resolveSize(preferredHeight, heightMeasureSpec)

        setMeasuredDimension(finalWidth, finalHeight)
    }

    override fun onDraw(canvas: Canvas) {
        drawChipBackground(canvas)
        drawText(canvas)
    }

    private fun drawChipBackground(canvas: Canvas) {
        val chipLeft = paddingLeft
        val chipTop = paddingTop
        val chipRight = width - paddingRight
        val chipBottom = height - paddingBottom

        chipBackgroundDrawable.apply {
            setBounds(chipLeft, chipTop, chipRight, chipBottom)
            draw(canvas)
        }
    }

    private fun drawText(canvas: Canvas) {
        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it)
            val textHeight = textMeasuringRect.height()
            val clientWidth = width - paddingLeft - paddingRight - chipPaddingLeft - chipPaddingRight
            val clientHeight = height - paddingTop - paddingBottom - chipPaddingTop - chipPaddingBottom
            val textX = paddingLeft + chipPaddingLeft + (clientWidth - textWidth) / 2f
            val textY = paddingTop + chipPaddingTop - textMeasuringRect.top + (clientHeight - textHeight) / 2f

            canvas.drawText(it, textX, textY, textPaint)
        }
    }
}