package com.apandroid.chiptextview

import android.content.Context
import android.content.res.TypedArray
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.text.TextPaint
import android.util.AttributeSet
import android.view.View
import kotlin.math.max
import kotlin.math.roundToInt

class ChipTextView(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : View(context, attrs, defStyleAttr) {

    private val textMeasuringRect = Rect()
    private val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
    private var chipBackgroundDrawable = ChipBackgroundDrawable()

    var text: String? = null
        set(value) {
            field = value
            requestLayout()
        }

    var textSize
        get() = textPaint.textSize
        set(value) {
            textPaint.textSize = value
            requestLayout()
        }

    var textColor
        get() = textPaint.color
        set(value) {
            textPaint.color = value
            invalidate()
        }

    var chipPaddingLeft = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingRight = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingTop = 0
        set(value) {
            field = value
            requestLayout()
        }

    var chipPaddingBottom = 0
        set(value) {
            field = value
            requestLayout()
        }

    val chipDrawable get() = chipBackgroundDrawable

    init {
        parseAttributes(context, attrs, defStyleAttr)
    }

    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, R.attr.chipTextViewStyle)

    constructor(context: Context) : this(context, null)

    private fun parseAttributes(context: Context, attrs: AttributeSet?, defStyleAttr: Int) {
        attrs?.let {
            val array = context.obtainStyledAttributes(attrs, R.styleable.ChipTextView, defStyleAttr, R.style.ChipTextViewDefaultStyle)
            readText(array)
            readTextSize(array)
            readTextColor(array)
            readChipColor(array)
            readChipPadding(array)
            array.recycle()
        }
    }

    private fun readText(typedArray: TypedArray) {
        text = typedArray.getString(R.styleable.ChipTextView_ctv_text)
    }

    private fun readTextSize(typedArray: TypedArray) {
        textPaint.textSize = typedArray.getDimension(R.styleable.ChipTextView_ctv_textSize, 0f)
    }

    private fun readTextColor(typedArray: TypedArray) {
        textPaint.color = typedArray.getColor(R.styleable.ChipTextView_ctv_textColor, Color.WHITE)
    }

    private fun readChipColor(typedArray: TypedArray) {
        setChipColor(typedArray.getColor(R.styleable.ChipTextView_ctv_chipColor, Color.BLACK))
    }

    private fun readChipPadding(typedArray: TypedArray) {
        typedArray.apply {
            val padding = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPadding, 0)
            val paddingHorizontal = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingHorizontal, padding)
            val paddingVertical = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingVertical, padding)

            setChipPadding(padding)
            setChipHorizontalPadding(paddingHorizontal)
            setChipVerticalPadding(paddingVertical)

            chipPaddingLeft = readAliasDimension(typedArray, R.styleable.ChipTextView_ctv_chipPaddingLeft, R.styleable.ChipTextView_ctv_chipPaddingStart, paddingHorizontal)
            chipPaddingRight = readAliasDimension(typedArray, R.styleable.ChipTextView_ctv_chipPaddingRight, R.styleable.ChipTextView_ctv_chipPaddingEnd, paddingHorizontal)
            chipPaddingTop = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingTop, paddingVertical)
            chipPaddingBottom = getDimensionPixelSize(R.styleable.ChipTextView_ctv_chipPaddingBottom, paddingVertical)
        }
    }

    private fun readAliasDimension(typedArray: TypedArray, originName: Int, alias: Int, defaultValue: Int) = typedArray.run {
        if (hasValue(alias)) {
            getDimensionPixelSize(alias, 0)
        } else {
            getDimensionPixelSize(originName, defaultValue)
        }
    }

    fun setChipPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipHorizontalPadding(padding: Int) {
        chipPaddingLeft = padding
        chipPaddingRight = padding
    }

    fun setChipVerticalPadding(padding: Int) {
        chipPaddingTop = padding
        chipPaddingBottom = padding
    }

    fun setChipPadding(left: Int, top: Int, right: Int, bottom: Int) {
        chipPaddingLeft = left
        chipPaddingTop = top
        chipPaddingRight = right
        chipPaddingBottom = bottom
    }

    fun setChipColor(color: Int) {
        chipBackgroundDrawable = ChipBackgroundDrawable(color)
        invalidate()
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)

        var preferredWidth= 0
        var preferredHeight = 0

        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it).roundToInt()
            val textHeight = textMeasuringRect.height()

            if (it.length == 1) {
                val max = max(textWidth, textHeight)
                preferredWidth = max
                preferredHeight = max
            } else {
                preferredWidth = textWidth
                preferredHeight = textHeight
            }

            preferredWidth += paddingLeft + paddingRight + chipPaddingLeft + chipPaddingRight
            preferredHeight += paddingTop + paddingBottom + chipPaddingTop + chipPaddingBottom
        }

        val finalWidth = View.resolveSize(preferredWidth, widthMeasureSpec)
        val finalHeight = View.resolveSize(preferredHeight, heightMeasureSpec)

        setMeasuredDimension(finalWidth, finalHeight)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        drawChipBackground(canvas)
        drawText(canvas)
    }

    private fun drawChipBackground(canvas: Canvas) {
        val chipLeft = paddingLeft
        val chipTop = paddingTop
        val chipRight = width - paddingRight
        val chipBottom = height - paddingBottom

        chipBackgroundDrawable.apply {
            setBounds(chipLeft, chipTop, chipRight, chipBottom)
            draw(canvas)
        }
    }

    private fun drawText(canvas: Canvas) {
        text?.let {
            textPaint.getTextBounds(it, 0, it.length, textMeasuringRect)

            val textWidth = textPaint.measureText(it)
            val textHeight = textMeasuringRect.height()
            val clientWidth = width - paddingLeft - paddingRight - chipPaddingLeft - chipPaddingRight
            val clientHeight = height - paddingTop - paddingBottom - chipPaddingTop - chipPaddingBottom
            val textX = paddingLeft + chipPaddingLeft + (clientWidth - textWidth) / 2f
            val textY = paddingTop + chipPaddingTop - textMeasuringRect.top + (clientHeight - textHeight) / 2f

            canvas.drawText(it, textX, textY, textPaint)
        }
    }
}