/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */
package com.adobe.pdfservices.operation.pdfops.options.pdfproperties;

import com.adobe.pdfservices.operation.pdfops.PDFPropertiesOperation;


/**
 * Parameters for getting properties of a PDF using {@link PDFPropertiesOperation}.
 */
public class PDFPropertiesOptions {
    private boolean includePageLevelProperties;

    private PDFPropertiesOptions(Builder builder)
    {
        this.includePageLevelProperties = builder.includePageLevelProperties;
    }

    /**
     * Returns {@code true} if page level properties are set to be included in the resulting PDF properties.
     * Default value is {@code false}
     *
     * @return {@code true} if page level properties are set to be included in the resulting PDF properties.
     */
    public boolean includePageLevelProperties() {
        return includePageLevelProperties;
    }

    /**
     * Creates a new {@code PDFPropertiesOptions} builder.
     *
     * @return a {@code PDFPropertiesOptions.Builder} instance
     */
    public static PDFPropertiesOptions.Builder PDFPropertiesOptionsBuilder()
    {
        return new PDFPropertiesOptions.Builder();
    }

    /**
     * Builds a {@link PDFPropertiesOptions} instance.
     */
    public static class Builder
    {
        private boolean includePageLevelProperties;

        /**
         * Constructs a {@code Builder} instance.
         */
        private Builder() {}


        /**
         * Sets the includePageLevelProperties parameter.
         * If true, the page level properties of the input PDF will be included in the resulting JSON file or JSON Object.
         *
         * @param includePageLevelProperties true if page level properties should be included in the resulting PDF properties. Default value is {@code false}
         * @return this Builder instance to add any additional parameters
         */
        public Builder includePageLevelProperties(boolean includePageLevelProperties)
        {
            this.includePageLevelProperties = includePageLevelProperties;
            return this;
        }

        /**
         * Returns a new {@link PDFPropertiesOptions} instance built from the current state of this builder.
         *
         * @return a new {@code PDFPropertiesOptions} instance
         */
        public PDFPropertiesOptions build()
        {
            return new PDFPropertiesOptions(this);
        }
    }

}
