/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops.options.createpdf;

import com.adobe.pdfservices.operation.pdfops.CreatePDFOperation;
import org.json.JSONObject;

/**
 * Parameters for converting HTML to PDF using {@link CreatePDFOperation}.
 */
public class CreatePDFFromHTMLOptions extends CreatePDFOptions {


    /**
     * Default values for options
     */
    private static final boolean PRINT_HEADER_FOOTER_BY_DEFAULT = true;

    private boolean includeHeaderFooter;

    private PageLayout pageLayout;

    private JSONObject dataToMerge;

    private CreatePDFFromHTMLOptions(boolean includeHeaderFooter,
                                     PageLayout pageLayout,
                                     JSONObject dataToMerge) {
        this.includeHeaderFooter = includeHeaderFooter;
        this.pageLayout = pageLayout;
        this.dataToMerge = dataToMerge;
    }

    /**
     * Returns {@code true} if default header and footer will be included in the resulting PDF file.
     * The default header consists of the date and the document.title.
     * The default footer consists of the file name and page number.
     *
     * @return {@code true} if default header and footer will be included in the resulting PDF file
     */
    public boolean includesHeaderFooter() {
        return includeHeaderFooter;
    }

    /**
     * Returns the intended page layout of the resulting PDF file.
     *
     * @return intended page layout
     */
    public PageLayout getPageLayout() {
        return pageLayout;

    }

    /**
     * Returns JSON data that will be used to manipulate HTML DOM before it is converted into PDF file.
     * This mechanism is intended to be used to supply data that might otherwise be retrieved using ajax requests.
     * <p>
     * To make use of this mechanism, the source html file must include a script element such as:
     * <pre>{@code <script src='./json.js' type='text/javascript'></script>}
     *      where json.js refers to the JSON data,</pre>
     * And also Requires javascript in the source html file to make use of this JSON data to manipulate the HTML DOM.
     *
     * @return data inputs for manipulating HTML DOM before converting it into a PDF file.
     */
    public JSONObject getDataToMerge() {
        return dataToMerge;

    }

    /**
     * Builds a {@link CreatePDFFromHTMLOptions} instance.
     */
    public static class Builder {
        private boolean includeHeaderFooter;
        private PageLayout pageLayout;
        private JSONObject dataToMerge;

        /**
         * Constructs a {@code Builder} instance.
         */
        public Builder() {
            this.includeHeaderFooter = PRINT_HEADER_FOOTER_BY_DEFAULT;
            this.pageLayout = new PageLayout();
            this.dataToMerge = new JSONObject();
        }

        /**
         * Sets the includeHeaderFooter parameter. If true, default header and footer will be included in resulting PDF.
         * The default header consists of the date and the document.title.
         * The default footer consists of the file name and page number.
         *
         * @param includeHeaderFooter true if default header and footer should be included in the resulting PDF. Default value is {@code true}
         * @return this Builder instance to add any additional parameters
         */
        public Builder includeHeaderFooter(boolean includeHeaderFooter) {
            this.includeHeaderFooter = includeHeaderFooter;
            return this;
        }

        /**
         * Sets the pageLayout parameter.
         *
         * @param pageLayout intended page layout of the resulting PDF file
         * @return this Builder instance to add any additional parameters
         */
        public Builder withPageLayout(PageLayout pageLayout) {
            this.pageLayout = pageLayout;
            return this;
        }

        /**
         * Sets the data to be used by the javascript in the source html file to manipulate the HTML DOM before it gets converted to PDF.
         * This mechanism is intended to be used to supply data that might otherwise be retrieved using ajax requests.
         * <p>
         * To make use of this mechanism, the source html file must include a script element such as:
         * <pre>{@code <script src='./json.js' type='text/javascript'></script>}
         *      where json.js refers to the JSON data,</pre>
         * And also Requires javascript in the source html file to make use of this JSON data to manipulate the HTML DOM.
         *
         * @param dataToMerge JSON object
         * @return this Builder instance to add any additional parameters
         */
        public Builder withDataToMerge(JSONObject dataToMerge) {
            this.dataToMerge = dataToMerge;
            return this;
        }

        /**
         * Returns a new {@link CreatePDFFromHTMLOptions} instance built from the current state of this builder.
         *
         * @return a new {@code CreatePDFFromHTMLOptions} instance
         */
        public CreatePDFFromHTMLOptions build() {
            return new CreatePDFFromHTMLOptions(includeHeaderFooter, pageLayout, dataToMerge);

        }
    }
}
