/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.adobe.pdfservices.operation.exception.ServiceApiException;
import com.adobe.pdfservices.operation.exception.ServiceUsageException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.MediaType;
import com.adobe.pdfservices.operation.internal.api.FileDownloadApi;
import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.CPFContentAnalyzerResponse;
import com.adobe.pdfservices.operation.internal.exception.OperationException;
import com.adobe.pdfservices.operation.internal.service.DocumentMergeService;
import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.internal.util.ValidationUtil;
import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.Operation;
import com.adobe.pdfservices.operation.io.FileRef;
import com.adobe.pdfservices.operation.pdfops.options.documentmerge.DocumentMergeOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An operation that enables the clients to produce high fidelity PDF and Word documents with dynamic data inputs.
 * This operation merges the JSON data with the Word template to create dynamic documents for contracts and
 * agreements, invoices, proposals, reports, forms, branded marketing documents and more.
 * <p>
 * To know more about document generation and document templates, please see the <a href="http://www.adobe.com/go/dcdocgen_overview_doc" target="_blank">documentation</a>
 * <p>
 * Sample Usage:
 * <pre>{@code   JSONObject jsonDataForMerge = new JSONObject("{\"customerName\": \"James\",\"customerVisits\": 100}");
 *   DocumentMergeOptions documentMergeOptions = new DocumentMergeOptions(jsonDataForMerge, OutputFormat.PDF);
 *   DocumentMergeOperation docMergeOperation = DocumentMergeOperation.createNew(documentMergeOptions);
 *   documentMergeOptions.setInput(FileRef.createFromLocalFile("~/Documents/documentTemplate.docx",
 *                                                           DocumentMergeOptions.SupportedSourceFormat.DOCX.getMediaType()));
 *   Credentials credentials = Credentials.serviceAccountCredentialsBuilder().fromFile("pdfservices-api-credentials.json").build();
 *   FileRef result = createPdfOperation.execute(ExecutionContext.create(credentials));
 *   result.saveAs("output/DocumentMergeOutput.pdf");
 * }</pre>
 */
public class DocumentMergeOperation implements Operation {

    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentMergeOperation.class);

    /**
     * Variable to refer input document template for the DocumentMergeOperation
     */
    private FileRefImpl documentTemplate;

    /**
     * Variable to provide Options for DocumentMergeOperation
     */
    private DocumentMergeOptions documentMergeOptions;

    /**
     * Supported media types for this operation
     */
    private static final Set<String> SUPPORTED_SOURCE_MEDIA_TYPE =
            new HashSet<>(Arrays.asList(ExtensionMediaTypeMapping.DOCX.getMediaType()));


    private DocumentMergeOperation(DocumentMergeOptions documentMergeOptions) {
        this.documentMergeOptions = documentMergeOptions;
    }

    /**
     * Sets the input MS Word based document template
     *
     * @param documentTemplate an input file; Cannot be null
     */
    public void setInput(FileRef documentTemplate) {
        Objects.requireNonNull(documentTemplate, "No input was set for operation");
        this.documentTemplate = (FileRefImpl) documentTemplate;
    }

    /**
     * Variable to check if the operation instance was invoked more than once
     */
    private boolean isInvoked = false;

    /**
     * Constructs a {@code DocumentMergeOperation} instance.
     *
     * @param  documentMergeOptions for merging document template with provided input JSON data; Cannot be null
     * @return a new {@code DocumentMergeOperation} instance
     */
    public static DocumentMergeOperation createNew(DocumentMergeOptions documentMergeOptions) {
        return new DocumentMergeOperation(documentMergeOptions);
    }

    /**
     * Executes this operation synchronously using the supplied context and returns a new FileRef instance for the resulting Merged file.
     * <p>
     * The resulting file may be stored in the system temporary directory (per java.io.tmpdir System property).
     * See {@link FileRef} for how temporary resources are cleaned up.
     *
     * @param context the context in which to execute the operation
     * @return the resulting merged document
     * @throws ServiceApiException   if an API call results in an error response
     * @throws IOException           if there is an error in reading either the input source or the resulting PDF file
     * @throws ServiceUsageException if service usage limits have been reached or credentials quota has been exhausted
     */
    public FileRef execute(ExecutionContext context) throws ServiceApiException, IOException, ServiceUsageException {
        validateInvocationCount();
        InternalExecutionContext internalExecutionContext = (InternalExecutionContext) context;
        this.validate(internalExecutionContext);

        try {
            LOGGER.info("All validations successfully done. Beginning DocumentMerge operation execution");
            long startTimeMs = System.currentTimeMillis();

            String location = DocumentMergeService.mergeDocument(internalExecutionContext,
                    documentTemplate, documentMergeOptions, this.getClass().getSimpleName());

            String targetFileName = FileUtil.getRandomFileName(documentMergeOptions.getOutputFormat().getFormat());
            String temporaryDestinationPath = PathUtil.getTemporaryDestinationPath(targetFileName, documentMergeOptions.getOutputFormat().getFormat());


            FileDownloadApi.downloadAndSave(internalExecutionContext, location, temporaryDestinationPath, CPFContentAnalyzerResponse.class);
            LOGGER.info("Operation successfully completed. Stored created File at {}", temporaryDestinationPath);
            LOGGER.debug("Operation Success Info - Request ID: {}, Latency(ms): {}",
                    StringUtil.getRequestIdFromLocation(location), System.currentTimeMillis() - startTimeMs);

            isInvoked = true;
            return FileRef.createFromLocalFile(temporaryDestinationPath);
        } catch (OperationException oe) {
            throw new ServiceApiException(oe.getErrorMessage(), oe.getRequestTrackingId(), oe.getStatusCode(), oe.getReportErrorCode());
        }

    }

    private void validate(InternalExecutionContext context) {
        if (documentTemplate == null) {
            throw new IllegalArgumentException("No input was set for operation");
        }
        if (documentTemplate.getSourceURL() != null) {
            throw new IllegalArgumentException("Input for the Document Merge Operation can not be sourced from a URL");
        }
        ValidationUtil.validateExecutionContext(context);
        ValidationUtil.validateMediaType(SUPPORTED_SOURCE_MEDIA_TYPE, this.documentTemplate.getMediaType());
        Objects.requireNonNull(documentMergeOptions, "Document Merge Options cannot be null");
        ValidationUtil.validateDocumentMergeOptions(documentMergeOptions);
    }


    private void validateInvocationCount() {
        if (isInvoked) {
            LOGGER.error("Operation instance must only be invoked once");
            throw new IllegalStateException("Operation instance must not be reused, can only be invoked once");
        }
    }

    /**
     * Supported source file formats for {@link DocumentMergeOperation}.
     */
    public enum SupportedSourceFormat implements MediaType {
        /**
         * Represents "application/vnd.openxmlformats-officedocument.wordprocessingml.document" media type
         */
        DOCX;

        /**
         * Returns the corresponding media type for this format, intended to be used for {@code mediaType} parameter in
         * {@link FileRef} methods.
         *
         * @return the corresponding media type
         */
        public String getMediaType() {
            return ExtensionMediaTypeMapping.valueOf(name()).getMediaType().toLowerCase();
        }
    }


}
