/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.util;


import java.io.File;

import com.adobe.pdfservices.operation.internal.GlobalConfig;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PathUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(PathUtil.class);

    private static void createTempDirectoryIfRequired() {
        File file = new File(getTemporaryDirectoryPath());
        if (!file.exists()) {
            file.mkdir();
        } else {
            LOGGER.debug("Temporary directory already exists. Won't be created again");
        }
    }

    /**
     * Utility method to get the directory path of the folder where the operation results are stored temporarily
     *
     * @return
     */
    public static String getTemporaryDirectoryPath() {
        String systemTemporaryDirectory = GlobalConfig.getSystemTemporaryDirectory();
        if (systemTemporaryDirectory.endsWith(File.separator)) {
            return String.format("%s%s", systemTemporaryDirectory,
                    GlobalConfig.getTemporaryOperationResultDirectory());
        } else {
            return String.format("%s%s%s", systemTemporaryDirectory, File.separator, GlobalConfig.getTemporaryOperationResultDirectory());
        }
    }

    /**
     * Utility method to get a temporary destination path for a given file with a target format
     *
     * @param sourceName      name of the file
     * @param targetExtension target format of the file, must be only the format characters without any '.'
     * @return
     */
    public static String getTemporaryDestinationPath(String sourceName, String targetExtension) {
        String destinationFileName;
        if (StringUtil.isBlank(sourceName)) {
            destinationFileName = FileUtil.getRandomFileName(targetExtension);
        } else {
            destinationFileName = getFileNameWithExtension(getBaseName(sourceName), targetExtension);
        }

        createTempDirectoryIfRequired();
        return String.format("%s%s%s", getTemporaryDirectoryPath(), File.separator, destinationFileName);
    }

    public static String getFileNameWithExtension(String baseName, String targetExtension) {
        if (targetExtension == null) {
            return baseName;
        }
        if (targetExtension.startsWith(".")) {
            return baseName + targetExtension;
        }
        return baseName + "." + targetExtension;
    }

    public static String getExtension(String filePath) {
        return FilenameUtils.getExtension(filePath);
    }

    public static String getFileName(String filePath) {
        if (StringUtil.isNotBlank(filePath)) {
            return FilenameUtils.getName(filePath);
        }
        return "";
    }

    public static String getBaseName(String filePath) {
        if (StringUtil.isNotBlank(filePath)) {
            return FilenameUtils.getBaseName(filePath);
        }
        return "";
    }

    public static String getFullPath(String filePath) {
        if (StringUtil.isNotBlank(filePath)) {
            return FilenameUtils.getFullPath(filePath);
        }
        return "";
    }

    public static String getSubPath(String directory, String subDirectory) {
        if (StringUtil.isNotBlank(directory) && StringUtil.isNotBlank(subDirectory)) {
            return directory + File.separator + subDirectory;
        }
        return "";
    }


}
