/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.service;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import com.adobe.pdfservices.operation.internal.cpf.constants.CPFConstants;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.CPFContentAnalyzerRequests;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.ExportPDFParams;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.Inputs;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.api.CPFApi;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.Outputs;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.pdfops.options.exportpdf.ExportPDFTargetFormat;
import com.adobe.pdfservices.operation.pdfops.options.exportpdftoimages.ExportPDFToImagesTargetFormat;

public class ExportPDFService {

    public static String exportPDF(InternalExecutionContext context,
                                   FileRefImpl sourceFileRef,
                                   ExportPDFTargetFormat exportPDFTargetFormat,
                                   String operation) throws FileNotFoundException {
        try {
            // Create ExportPDFParams
            ExportPDFParams exportPDFParams = ExportPDFParams.createFrom(exportPDFTargetFormat);

            // Create cpf inputs from ExportPDFParams
            Inputs inputs = Inputs.builder(sourceFileRef.getMediaType())
                    .setParams(exportPDFParams)
                    .build();

            // Create CPFContentAnalyzerRequests from cpf inputs and cpf outputs
            CPFContentAnalyzerRequests cpfContentAnalyzerRequests
                    = new CPFContentAnalyzerRequests(CPFConstants.ExportPDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, getOutputFormat(exportPDFTargetFormat));

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = new ArrayList<>();
            sourceFileRefList.add(sourceFileRef);

            HttpResponse<String> response = CPFApi.cpfCreateOpsApi(context, cpfContentAnalyzerRequests,
                    sourceFileRefList, String.class, operation);

            return response.getHeaders().get(DefaultRequestHeaders.LOCATION_HEADER_NAME);
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    public static String exportPDFToImages(InternalExecutionContext context,
                                   FileRefImpl sourceFileRef,
                                   ExportPDFToImagesTargetFormat exportPDFToImagesTargetFormat,
                                   String operation) throws FileNotFoundException {
        try {
            // Create ExportPDFParams
            ExportPDFParams exportPDFParams = ExportPDFParams.createImagesFrom(exportPDFToImagesTargetFormat);

            // Create cpf inputs from ExportPDFParams
            Inputs inputs = Inputs.builder(sourceFileRef.getMediaType())
                    .setParams(exportPDFParams)
                    .build();

            Outputs outputs = new Outputs(exportPDFToImagesTargetFormat.getFileExt(), true);

            // Create CPFContentAnalyzerRequests from cpf inputs and cpf outputs
            CPFContentAnalyzerRequests cpfContentAnalyzerRequests
                    = new CPFContentAnalyzerRequests(CPFConstants.ExportPDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, outputs);

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = new ArrayList<>();
            sourceFileRefList.add(sourceFileRef);

            HttpResponse<String> response = CPFApi.cpfCreateOpsApi(context, cpfContentAnalyzerRequests,
                    sourceFileRefList, String.class, operation);

            return response.getHeaders().get(DefaultRequestHeaders.LOCATION_HEADER_NAME);
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    private static String getOutputFormat(ExportPDFTargetFormat exportPDFTargetFormat) {
        switch (exportPDFTargetFormat) {
            case PNG:
            case JPEG:
                return ExtensionMediaTypeMapping.ZIP.getMediaType();
            default:
                return ExtensionMediaTypeMapping
                        .getFromExtension(exportPDFTargetFormat.getFileExt()).getMediaType();
        }
    }
}
