/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.http;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.mail.MessagingException;
import javax.mail.internet.ContentDisposition;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.mail.util.ByteArrayDataSource;

import org.apache.commons.collections4.CollectionUtils;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.MultiPartFormField;
import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.CPFContentAnalyzerResponse;
import com.adobe.pdfservices.operation.internal.util.JsonUtil;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultiPartHttpResponse<T> implements HttpResponse<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(MultiPartHttpResponse.class);

    private static final String CONTENT_TYPE_MULTIPART_MIXED_STRING = "multipart/mixed";
    private static final String MIME_TYPE_APPLICATION_JSON_STRING = "application/json";
    private static final String MIME_TYPE_APPLICATION_OCTET_STREAM_STRING = "application/octet-stream";
    private static final String CONTENT_DISPOSITION_STRING = "Content-Disposition";
    private static final String NAME_CONTENT_DISPOSITION_PARAM_STRING = "name";
    private static final String FILE_OUTPUT_LABEL_STRING = "file";

    private int statusCode;
    private Map<String, String> headers;
    private T inlineJsonResponseDto;
    private Class<T> inlineJsonResponseType;
    private List<InputStream> inlineContentResponses;
    private List<MimeBodyPart> responseBodyParts;


    public MultiPartHttpResponse(int statusCode, Map<String,String> headers,
                                 InputStream inputStream,
                                 Class<T> inlineJsonResponseType) throws IOException, MessagingException {
        this.statusCode = statusCode;
        this.headers = headers;
        this.inlineJsonResponseType = inlineJsonResponseType;
        this.inlineContentResponses = new ArrayList<>();
        parseInlineResponseData(inputStream);
    }

    private void parseInlineResponseData(InputStream inputStream) throws IOException, MessagingException {

        ByteArrayDataSource datasource =
                new ByteArrayDataSource(IOUtils.toByteArray(inputStream), CONTENT_TYPE_MULTIPART_MIXED_STRING);
        MimeMultipart multipart = new MimeMultipart(datasource);
        this.responseBodyParts = new ArrayList<MimeBodyPart>();

        Map<String, InputStream> fileNameToStreamMap = new HashMap<>();
        int count = multipart.getCount();
        for (int i = 0; i < count; i++) {
            MimeBodyPart bodyPart = (MimeBodyPart) multipart.getBodyPart(i);
            responseBodyParts.add(bodyPart);
            if (bodyPart.isMimeType(MIME_TYPE_APPLICATION_JSON_STRING)) {
                processJsonData(bodyPart.getInputStream());
            } else if (bodyPart.isMimeType(MIME_TYPE_APPLICATION_OCTET_STREAM_STRING)) {
                processBinaryData(fileNameToStreamMap, bodyPart);
            }
        }

        /*
        Following logic makes sure that in case of multiple output files, the order of output files follows the order
        specified by the CPFContentAnalyzerResponse's documentOutList
         */
        // case for CPFContentAnalyzerResponse
        if (inlineJsonResponseDto instanceof CPFContentAnalyzerResponse) {
            CPFContentAnalyzerResponse cpfContentAnalyzerResponse = (CPFContentAnalyzerResponse) inlineJsonResponseDto;
            List<MultiPartFormField> documentOutList
                    = cpfContentAnalyzerResponse.getCpfOutputs() != null ? cpfContentAnalyzerResponse.getCpfOutputs().getDocumentOutList(): null;

            // if output is a list of docs
            if (documentOutList != null && documentOutList.size() > 0) {
                for(MultiPartFormField multiPartFormField : documentOutList) {
                    inlineContentResponses.add(fileNameToStreamMap.get(multiPartFormField.getLocation()));
                }
            // Only single output expected when there is no documentOutList
            } else {
                inlineContentResponses.add(fileNameToStreamMap.get(FILE_OUTPUT_LABEL_STRING));
            }
        } else {
            inlineContentResponses.add(fileNameToStreamMap.get(FILE_OUTPUT_LABEL_STRING));
        }
        inputStream.close();
    }

    @Override
    public Map<String, String> getHeaders() {
        return headers;
    }

    @Override
    public int getStatusCode() {
        return statusCode;
    }

    @Override
    public String getRequestId() {
        if (headers != null) {
            return headers.get(DefaultRequestHeaders.DC_REQUEST_ID_HEADER_KEY);
        }
        return null;
    }

    @Override
    public T getBody() {
        return inlineJsonResponseDto;
    }

    @Override
    public List<InputStream> getResponseAsStreamList() {
        return inlineContentResponses;
    }

    @Override
    public void consume() throws IOException {
        for(InputStream inlineContentResponse : inlineContentResponses) {
            if (inlineContentResponse != null) {
                inlineContentResponse.close();
            }
        }
        if (CollectionUtils.isNotEmpty(responseBodyParts)) {
            for (MimeBodyPart responseBodyPart: responseBodyParts) {
                try {
                    responseBodyPart.getInputStream().close();
                } catch (Exception ex) {
                    continue;
                }
            }
        }
    }

    private void processJsonData(InputStream inputStream) {
        this.inlineJsonResponseDto = JsonUtil.deserializeJsonStream(inputStream, this.inlineJsonResponseType);
    }

    private void processBinaryData(Map<String, InputStream> fileNameToStreamMap, MimeBodyPart bodyPart) throws IOException, MessagingException {
        ContentDisposition contentDisposition
                = new ContentDisposition(bodyPart.getHeader(CONTENT_DISPOSITION_STRING, ":"));
        String name = contentDisposition.getParameter(NAME_CONTENT_DISPOSITION_PARAM_STRING);
        fileNameToStreamMap.put(name, bodyPart.getInputStream());
    }

    public List<MimeBodyPart> getResponseBodyParts() {
        return responseBodyParts;
    }

}
