/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.http;


import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.exception.SdkException;
import com.adobe.pdfservices.operation.internal.auth.AuthenticationMethod;
import com.adobe.pdfservices.operation.internal.auth.Authenticator;
import com.adobe.pdfservices.operation.internal.auth.SessionToken;

import com.github.hal4j.uritemplate.URITemplate;
import org.apache.http.Consts;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.entity.StringEntity;

public class BaseHttpRequest implements HttpRequest {

    private URITemplate uriTemplate;
    private HttpMethod httpMethod;
    private RequestType requestType;
    private Map<String, String> headers;
    private AuthenticationMethod authenticationMethod;
    private StringEntity entity;
    private Authenticator authenticator;
    private RequestKey requestKey;
    /**
     * Config which deals with http timeouts and request specific params.
     */
    private HttpRequestConfig requestConfig;

    public BaseHttpRequest(String requestKey, HttpMethod httpMethod, String uriTemplate) {
        Objects.requireNonNull(requestKey);
        Objects.requireNonNull(httpMethod);
        Objects.requireNonNull(uriTemplate);
        this.httpMethod = httpMethod;
        this.uriTemplate = new URITemplate(uriTemplate);
        this.headers = new HashMap<>();
        this.authenticationMethod = AuthenticationMethod.AUTH_HEADER_PRIMARY;
        this.requestType = RequestType.REGULAR;
        this.requestKey = RequestKey.getRequestKeyFromValue(requestKey);
    }

    public BaseHttpRequest(HttpMethod httpMethod) {
        Objects.requireNonNull(httpMethod);
        this.httpMethod = httpMethod;
        this.headers = new HashMap<>();
        this.authenticationMethod = AuthenticationMethod.AUTH_HEADER_PRIMARY;
        this.requestType = RequestType.REGULAR;
        this.requestKey = null;
    }

    @Override
    public HttpRequest withContentType(String contentType) {
        return withHeader(HttpHeaders.CONTENT_TYPE, contentType);
    }

    @Override
    public HttpRequest withHeader(String headerName, String headerValue) {
        this.headers.put(headerName, headerValue);
        return this;
    }

    /**
     * Will overwrite headers as well
     *
     * @param headers
     * @return
     */
    @Override
    public HttpRequest withHeaders(Map<String, String> headers) {
        Map<String, String> headersCopy = getHeadersCopy(headers);
        this.headers.putAll(headersCopy);
        return this;
    }

    @Override
    public HttpRequest withUriParam(String name, String value) {
        return null;
    }

    @Override
    public HttpRequest withUrlEncodedFormParams(List<NameValuePair> formParams) {
        this.entity = new UrlEncodedFormEntity(formParams,Consts.UTF_8);
        return this;
    }

    @Override
    public HttpRequest withBody(String body) {
        try {
            this.entity = new StringEntity(body);
        } catch (UnsupportedEncodingException e) {
            throw new SdkException(String.format("Internal error in creating http request: %s",body), e);
        }
        return this;
    }

    @Override
    public HttpRequest withAuthenticationMethod(AuthenticationMethod authMethod) {
        this.authenticationMethod = authMethod;
        return this;
    }

    @Override
    public HttpRequest withAuthenticator(Authenticator authenticator) {
        this.authenticator = authenticator;
        this.headers.put(DefaultRequestHeaders.X_API_KEY_HEADER_NAME, this.authenticator.getClientId());
        return this;
    }

    @Override
    public HttpRequest withConfig(HttpRequestConfig requestConfig) {
        this.requestConfig = requestConfig;
        return this;
    }

    @Override
    public HttpRequest withRequestKey(RequestKey requestKey) {
        this.requestKey = requestKey;
        return this;
    }

    @Override
    public void authenticate() {
        if (authenticationMethod.isAuthRequired()) {
            if (authenticator == null) {
                throw new IllegalStateException("No authenticator provided for request. Cannot add authorization headers");
            }
            SessionToken sessionToken = this.authenticator.getSessionToken(requestConfig);
            withHeader(HttpHeaders.AUTHORIZATION, String.format("Bearer %s", sessionToken.getAccessToken()));
        }
    }

    @Override
    public void forceAuthenticate() {
        if (authenticationMethod.isAuthRequired()) {
            if (authenticator == null) {
                throw new IllegalStateException("No authenticator provided for request. Cannot add authorization headers");
            }
            SessionToken sessionToken = this.authenticator.refreshSessionToken(requestConfig);
            withHeader(HttpHeaders.AUTHORIZATION, String.format("Bearer %s", sessionToken.getAccessToken()));
        }
    }

    @Override
    public HttpRequestConfig getConfig() {
        return requestConfig;
    }

    @Override
    public HttpRequest withTemplate(String uriTemplate) {
        this.uriTemplate = new URITemplate(uriTemplate);
        return this;
    }

    @Override
    public HttpRequest withAcceptType(String acceptHeader) {
        return withHeader(HttpHeaders.ACCEPT, acceptHeader);
    }

    @Override
    public StringEntity getEntity() {
        return entity;
    }

    @Override
    public HttpMethod getHttpMethod() {
        return httpMethod;
    }


    @Override
    public AuthenticationMethod getAuthMethod() {
        return authenticationMethod;
    }

    @Override
    public Map<String, String> getHeaders() {
        return getHeadersCopy(this.headers);
    }

    @Override
    public String getFinalUri() {
        return this.uriTemplate.expand().toString();
    }

    @Override
    public String getTemplateString() {
        return uriTemplate.toString();
    }

    @Override
    public RequestType getRequestType() {
        return requestType;
    }

    @Override
    public Authenticator getAuthenticator() {
        return authenticator;
    }

    private Map<String, String> getHeadersCopy(Map<String, String> headersToCopy) {
        return headersToCopy.entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey,
                        Map.Entry::getValue));
    }

    @Override
    public RequestKey getRequestKey() {
        return requestKey;
    }
}
