/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.auth;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Set;
import java.util.stream.Collectors;
import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import javax.validation.constraints.NotBlank;

import com.adobe.pdfservices.operation.internal.auth.ServiceAccountCredentialsWithUri;
import com.adobe.pdfservices.operation.internal.util.JsonUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.io.FileUtils;

/**
 * Service Account credentials allow your application to call PDF Services API on behalf of the application itself, or on behalf of an enterprise organization.
 * For getting the credentials, <a href="https://www.adobe.com/go/dcsdks_credentials?ref=getStartedWithServicesSdk">Click Here</a>
 */
public class ServiceAccountCredentials extends Credentials {

    @NotBlank(message = "client_id must not be blank")
    private String clientId;

    @NotBlank(message = "client_secret must not be blank")
    private String clientSecret;

    @NotBlank(message = "private_key must not be blank")
    private String privateKey;

    @NotBlank(message = "organization id must not be blank")
    private String organizationId;

    @NotBlank(message = "account id must not be blank")
    private String accountId;

    protected ServiceAccountCredentials(String clientId, String clientSecret, String privateKey, String organizationId, String accountId) {
        super();
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.privateKey = privateKey;
        this.organizationId = organizationId;
        this.accountId = accountId;
    }

    /**
     * Client Id (API Key)
     */
    public String getClientId() {
        return clientId;
    }

    /**
     * Client Secret
     */
    public String getClientSecret() {
        return clientSecret;
    }

    /**
     * Content of the Private Key (PEM format)
     */
    public String getPrivateKey() {
        return privateKey;
    }

    /**
     * Identifies the organization(format: org_ident@AdobeOrg) that has been configured for access to PDF Services API.
     */
    public String getOrganizationId() {
        return organizationId;
    }

    /**
     * Account ID(format: id@techacct.adobe.com)
     */
    public String getAccountId() {
        return accountId;
    }

    protected void validate() {
        ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
        Validator validator = factory.getValidator();
        Set<ConstraintViolation<Object>> violations = validator.validate(this);
        if (violations != null && !violations.isEmpty()) {
            String message = violations.stream().map(ConstraintViolation::getMessage).collect(Collectors.joining("; "));
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Builds a {@link ServiceAccountCredentials} instance.
     */
    public static class Builder {

        private String clientId;
        private String clientSecret;
        // TODO provide two setters, one for direct content and other for file object. Discuss on mail.
        private String privateKey;
        private String organizationId;
        private String accountId;
        // TODO: Set claim to be default and decide if we want to keep the setters for imsBaseUri and claim for stage env property or just from File
        // Can't be exposed, so get rid of corresponding setters. Leave them in fromFile
        private String imsBaseUri;
        private String claim;
        private String cpfOpsCreateUri;

        private static final String CLIENT_CREDENTIALS = "client_credentials";
        private static final String CLIENT_ID = "client_id";
        private static final String CLIENT_SECRET = "client_secret";

        private static final String SERVICE_ACCOUNT_CREDENTIALS = "service_account_credentials";
        // TODO set private key as file path instead of whole content
        private static final String PRIVATE_KEY_FILE = "private_key_file";
        private static final String CLAIM = "claim";
        private static final String ORGANIZATION_ID = "organization_id";
        private static final String TECHNICAL_ACCOUNT_ID = "account_id";
        private static final String IMS_BASE_URI = "ims_base_uri";
        private static final String CPF_SERVICES= "cpfServices";
        private static final String CPF_OPS_CREATE_URI= "cpfOpsCreateUri";

        /**
         * Constructs a {@code Builder} instance.
         */
        public Builder() {
        }

        /**
         * Set Client ID (API Key)
         * @param clientId
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder withClientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        /**
         * Set Client Secret
         * @param clientSecret
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder withClientSecret(String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        /**
         * Set private key
         * @param privateKey content of the private key file in PEM format
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder withPrivateKey(String privateKey) {
            this.privateKey = privateKey;
            return this;
        }

        /**
         * Set Organization Id (format: org_ident@AdobeOrg) that has been configured for access to PDF Services API
         * @param organizationId
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder withOrganizationId(String organizationId) {
            this.organizationId = organizationId;
            return this;
        }

        /**
         * Set Account Id (format: id@techacct.adobe.com)
         * @param accountId
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder withAccountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        /**
         * Sets Service Account Credentials using the JSON credentials file path.
         * All the keys in the JSON structure are optional.
         * <p>
         * JSON structure:
         * <pre>{
         *   "client_credentials": {
         *     "client_id": "CLIENT_ID",
         *     "client_secret": "CLIENT_SECRET"
         *   },
         *   "service_account_credentials": {
         *     "organization_id": "org_ident@AdobeOrg",
         *     "account_id": "id@techacct.adobe.com",
         *     "private_key_file": "private.key"
         *   }
         * }</pre>
         * private_key_file is the path of private key file. It will be looked up in the classpath and the directory of JSON credentials file.
         *
         * @return this Builder instance to add any additional parameters
         */
        public ServiceAccountCredentials.Builder fromFile(String credentialsFilePath) throws IOException {
            File credentialsFile = new File(credentialsFilePath);
            JsonNode credentialsConfig = JsonUtil.readJsonTreeFromFile(credentialsFile);
            JsonNode clientCredentials = credentialsConfig.get(CLIENT_CREDENTIALS);
            if(clientCredentials != null){
                JsonNode clientId = clientCredentials.get(CLIENT_ID);
                if(clientId != null && StringUtil.isNotBlank(clientId.asText())){
                    this.clientId = clientId.asText();
                }
                JsonNode clientSecret = clientCredentials.get(CLIENT_SECRET);
                if(clientSecret != null && StringUtil.isNotBlank(clientSecret.asText())){
                    this.clientSecret = clientSecret.asText();
                }
            }

            JsonNode serviceAccountCredentials = credentialsConfig.get(SERVICE_ACCOUNT_CREDENTIALS);
            if(serviceAccountCredentials != null){
                JsonNode organizationId = serviceAccountCredentials.get(ORGANIZATION_ID);
                if(organizationId != null && StringUtil.isNotBlank(organizationId.asText())){
                    this.organizationId = organizationId.asText();
                }
                JsonNode accountId = serviceAccountCredentials.get(TECHNICAL_ACCOUNT_ID);
                if(accountId != null && StringUtil.isNotBlank(accountId.asText())){
                    this.accountId = accountId.asText();
                }
                JsonNode privateKeyFileNode = serviceAccountCredentials.get(PRIVATE_KEY_FILE);
                if(privateKeyFileNode != null && StringUtil.isNotBlank(privateKeyFileNode.asText())){
                    File privateKeyFile = new File(privateKeyFileNode.asText());
                        if (!privateKeyFile.exists()) {
                            privateKeyFile = new File(credentialsFile.getParent() + File.separator + privateKeyFileNode.asText());
                        }
                        this.privateKey = FileUtils.readFileToString(privateKeyFile, StandardCharsets.UTF_8);
                }
                JsonNode imsBaseUri = serviceAccountCredentials.get(IMS_BASE_URI);
                if(imsBaseUri != null && StringUtil.isNotBlank(imsBaseUri.asText())){
                    this.imsBaseUri = imsBaseUri.asText();
                }
                JsonNode claim = serviceAccountCredentials.get(CLAIM);
                if(claim != null && StringUtil.isNotBlank(claim.asText())){
                    this.claim = claim.asText();
                }
                JsonNode cpfServices = credentialsConfig.get(CPF_SERVICES);
                if(cpfServices!=null) {
                    JsonNode cpfOpsCreateUri = cpfServices.get(CPF_OPS_CREATE_URI);
                    if (cpfOpsCreateUri != null && StringUtil.isNotBlank(cpfOpsCreateUri.asText())) {
                        this.cpfOpsCreateUri = cpfOpsCreateUri.asText();
                    }
                }
            }
            return this;
        }

        /**
         * Returns a new {@link ServiceAccountCredentials} instance built from the current state of this builder.
         *
         * @throws IllegalArgumentException if any of clientId, clientSecret, privateKey, accountId and organizationId are not provided.
         * @return a new {@code ServiceAccountCredentials} instance
         */
        public ServiceAccountCredentials build() {
            return new ServiceAccountCredentialsWithUri(imsBaseUri, clientId, clientSecret, privateKey, claim, organizationId, accountId, cpfOpsCreateUri);
        }
    }
}
