package com.aallam.openai.api.chat

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject

/**
 * Represents a function invocation with its name and serialized arguments as generated by the model.
 *
 * In scenarios such as a streaming variant of the chat API, both [nameOrNull] and [argumentsOrNull] can be null.
 */
@Serializable
public data class FunctionCall(
    /**
     * The name of the function to call.
     */
    @SerialName("name") val nameOrNull: String? = null,

    /**
     * The arguments to call the function with, as generated by the model in JSON format.
     * Note that the model does not always generate valid JSON, and may hallucinate parameters
     * not defined by your function schema.
     * Validate the arguments in your code before calling your function.
     */
    @SerialName("arguments") val argumentsOrNull: String? = null,
) {

    /**
     * The name of the function to call.
     */
    public val name: String
        get() = requireNotNull(nameOrNull)

    /**
     * The arguments to call the function with, as generated by the model in JSON format.
     * Note that the model does not always generate valid JSON, and may hallucinate parameters
     * not defined by your function schema.
     * Validate the arguments in your code before calling your function.
     */
    public val arguments: String
        get() = requireNotNull(argumentsOrNull)

    /**
     * Decodes the [arguments] JSON string into a JsonObject.
     * If [arguments] is null, the function will return null.
     *
     * @param json The Json object to be used for decoding, defaults to a default Json instance
     */
    public fun argumentsAsJson(json: Json = Json): JsonObject = json.decodeFromString(arguments)

    /**
     * Decodes the [arguments] JSON string into a JsonObject.
     * If [arguments] is null, the function will return null.
     *
     * @param json The Json object to be used for decoding, defaults to a default Json instance
     */
    public fun argumentsAsJsonOrNull(json: Json = Json): JsonObject? = argumentsOrNull?.let { json.decodeFromString(it) }
}
