package co.ronash.pushe.analytics.goal

import android.app.Activity
import android.content.Context
import android.support.v4.app.Fragment
import android.view.View
import android.widget.Button
import android.widget.Switch
import android.widget.TextView
import co.ronash.pushe.internal.PusheMoshi
import co.ronash.pushe.analytics.dagger.AnalyticsScope
import co.ronash.pushe.analytics.Constants
import co.ronash.pushe.analytics.GoalFragmentInfo
import co.ronash.pushe.analytics.LogTag.T_ANALYTICS
import co.ronash.pushe.analytics.LogTag.T_ANALYTICS_GOAL
import co.ronash.pushe.analytics.SessionFragmentInfo
import co.ronash.pushe.analytics.ViewExtractor
import co.ronash.pushe.analytics.utils.PreferenceConverter
import co.ronash.pushe.utils.log.Plog
import com.f2prateek.rx.preferences2.RxSharedPreferences
import com.squareup.moshi.Types
import java.util.concurrent.ConcurrentHashMap
import javax.inject.Inject

@AnalyticsScope
class GoalStore @Inject constructor(
    val context: Context,
    val moshi: PusheMoshi,
    val goalFragmentObfuscatedNameExtractor: GoalFragmentObfuscatedNameExtractor
) {
    private val sharedPreference = RxSharedPreferences
        .create(context.getSharedPreferences(GOAL_STORE_NAME, Context.MODE_PRIVATE))

    val definedGoals = sharedPreference.getObject (
        DEFINED_GOALS,
        mutableListOf<Goal>(),
        PreferenceConverter(moshi.adapter(Types.newParameterizedType(List::class.java, Goal::class.java)))
    )

    /**
     * A list containing all the [ViewGoalData]s for currently defined goals
     *
     * For the purpose of thread-safety, a [ConcurrentHashMap] is used instead of list.
     * The value for all the elements in the map is false and should be ignored
     *
     * Gets its data on initialize of the app
     * @see [extractViewGoalsDataSet]
     */
    var definedViewGoalsDataSet = ConcurrentHashMap<ViewGoalData, Boolean>()

    /**
     * A list containing all the [GoalData]s for currently defined goals
     *
     * For the purpose of thread-safety, a [ConcurrentHashMap] is used instead of list.
     * The value for all the elements in the map is false and should be ignored
     *
     * Gets its data on initialize of the app
     * @see [extractGoalsDataSet]
     */
    var definedGoalsDataSet = ConcurrentHashMap<GoalData, Boolean>()

    /**
     * must be called on initialize
     */
    fun initializeViewGoalsDataSet() {
        extractViewGoalsDataSet(definedGoals.get())
    }
    /**
     * must be called on initialize, after [initializeViewGoalsDataSet]
     */
    fun initializeGoalsDataSet() {
        extractGoalsDataSet(definedGoals.get())
    }

    /**
     * Called by [GoalProcessManager] when a newGoalMessage in received (@link [GoalProcessManager.updateGoals])
     *
     * Adds the new goals to [definedGoals] list
     * If there is already a goal with the same name, the goal will be replaced
     * ViewGoals of the new goals will be extracted and added to [definedViewGoalsDataSet]
     * GoalsDatas of the new goals will be built and added to [definedGoalsDataSet]
     *
     */
    fun updateGoals(goals: List<Goal>) {
        val definedGoals = definedGoals.get()
        for (goal in goals) {
            val currentGoals = definedGoals.filter {it.name == goal.name}
            if (currentGoals.isNotEmpty()) {
                definedGoals.remove(currentGoals[0])
                val goalData = definedGoalsDataSet.keys.filter { it.name == currentGoals[0].name }
                definedGoalsDataSet.remove(goalData[0])
                val goalViewGoalsDataSet = definedViewGoalsDataSet.keys.filter { it.parentGoalName == currentGoals[0].name }
                for (viewGoalData in goalViewGoalsDataSet){
                    definedViewGoalsDataSet.remove(viewGoalData)
                }
            }
            definedGoals.add(goal)
        }
        this.definedGoals.set(definedGoals)
        extractViewGoalsDataSet(goals)
        extractGoalsDataSet(goals)
        Plog[T_ANALYTICS, T_ANALYTICS_GOAL].info("New Goal Message received. Goals are updated"){
            "number of goals" to definedGoals.size
            "goals" to definedGoals
        }
    }

    /**
     * Called by [GoalProcessManager] when a removeGoalMessage in received (@link [GoalProcessManager.removeGoals])
     *
     * Removes the given goals from [definedGoals] list and [definedGoalsDataSet]
     * Removes the ViewGoalDatas of the given goals from [definedViewGoalsDataSet]
     *
     */
    fun removeGoals(goalNames: Set<String>) {
        val definedGoals = definedGoals.get()
        val goalsToBeRemoved = definedGoals.filter { goalNames.contains(it.name) }
        if (goalsToBeRemoved.size < goalNames.size){
            Plog[T_ANALYTICS, T_ANALYTICS_GOAL]
                .warn("removing defined Goals, some goals to be removed weren't found")
        }
        val goalsData = definedGoalsDataSet.keys.filter { goalNames.contains(it.name) }
        for (goalData in goalsData){
            definedGoalsDataSet.remove(goalData)
        }
        val viewGoalsDataSet = definedViewGoalsDataSet.keys.filter { goalNames.contains(it.parentGoalName) }
        for (viewGoalData in viewGoalsDataSet){
            definedViewGoalsDataSet.remove(viewGoalData)
        }

        for (goal in goalsToBeRemoved){
            definedGoals.remove(goal)
        }
        this.definedGoals.set(definedGoals)
    }

    /**
     * Called on initialize (@link [initializeViewGoalsDataSet]) and when new goals are defined (@link [updateGoals])
     *
     * Builds a [ViewGoalData] object for each viewGoal in the given goals and adds it to [definedViewGoalsDataSet]
     */
    private fun extractViewGoalsDataSet(goals: List<Goal>) {
        var viewGoalFragmentObfuscatedName: String?
        var goalGoalFragmentInfo: GoalFragmentInfo?
        for (goal in goals){
            for (viewGoal in goal.viewGoals) {
                if (viewGoal.goalMessageFragmentInfo == null) {
                    goalGoalFragmentInfo = null
                } else {
                    viewGoalFragmentObfuscatedName = goalFragmentObfuscatedNameExtractor.getFragmentObfuscatedName(viewGoal.goalMessageFragmentInfo)
                    goalGoalFragmentInfo = GoalFragmentInfo(
                        viewGoal.goalMessageFragmentInfo.actualName,
                        viewGoalFragmentObfuscatedName,
                        viewGoal.goalMessageFragmentInfo.fragmentId,
                        viewGoal.activityClassName
                    )
                }
                definedViewGoalsDataSet[ViewGoalData(
                        parentGoalName = goal.name,
                        targetValues = viewGoal.targetValues,
                        viewType = viewGoal.viewType,
                        viewID = viewGoal.viewID,
                        activityClassName = viewGoal.activityClassName,
                        goalFragmentInfo = goalGoalFragmentInfo)] = false
            }
        }
    }

    /**
     * Called on initialize (@link [initializeGoalsDataSet]) and when new goals are defined (@link [updateGoals])
     *
     * Builds a [GoalData] object for each Goal in the given list and adds it to [definedGoalsDataSet]
     */
    private fun extractGoalsDataSet(goals: List<Goal>) {
        for (goal in goals){
            when (goal.goalType) {
                GoalType.ACTIVITY_REACH -> {
                    definedGoalsDataSet[ActivityReachGoalData(
                        GoalType.ACTIVITY_REACH,
                        goal.name,
                        goal.activityClassName,
                        (goal as ActivityReachGoal).activityFunnel,
                        definedViewGoalsDataSet.keys.filter { it.parentGoalName == goal.name }
                    )] = false
                }
                GoalType.FRAGMENT_REACH -> {
                    definedGoalsDataSet[FragmentReachGoalData(
                        GoalType.FRAGMENT_REACH,
                        (goal as FragmentReachGoal).name,
                        goal.activityClassName,
                        GoalFragmentInfo(goal.goalMessageFragmentInfo.actualName,
                            goalFragmentObfuscatedNameExtractor.getFragmentObfuscatedName(goal.goalMessageFragmentInfo),
                            goal.goalMessageFragmentInfo.fragmentId,
                            goal.activityClassName),
                        goal.fragmentFunnel,
                        definedViewGoalsDataSet.keys.filter { it.parentGoalName == goal.name }
                    )] = false
                }
                GoalType.BUTTON_CLICK -> {
                    val goalMessageFragmentInfo = (goal as ButtonClickGoal).goalMessageFragmentInfo
                    val goalGoalFragmentInfo =
                        if (goalMessageFragmentInfo == null) null
                        else GoalFragmentInfo(goalMessageFragmentInfo.actualName,
                            goalFragmentObfuscatedNameExtractor.getFragmentObfuscatedName(goalMessageFragmentInfo),
                            goalMessageFragmentInfo.fragmentId,
                            goal.activityClassName)
                    definedGoalsDataSet[ButtonClickGoalData(
                        GoalType.BUTTON_CLICK,
                        goal.name,
                        goal.activityClassName,
                        goalGoalFragmentInfo,
                        goal.buttonID,
                        definedViewGoalsDataSet.keys.filter { it.parentGoalName == goal.name }
                    )] = false
                }
            }
        }
    }

    fun getActivityReachGoals(activityName: String): List<ActivityReachGoalData> {
        return definedGoalsDataSet.keys()
            .asSequence()
            .filter {
                it is ActivityReachGoalData &&
                        it.activityClassName == activityName
            }.map { it as ActivityReachGoalData }
            .toList()
    }

    fun getFragmentReachGoals(sessionFragmentInfo: SessionFragmentInfo): List<FragmentReachGoalData> {
        return definedGoalsDataSet.keys()
            .asSequence()
            .filter {
                it is FragmentReachGoalData &&
                        it.activityClassName == sessionFragmentInfo.activityName &&
                        (it.goalFragmentInfo.actualName == sessionFragmentInfo.fragmentName ||
                                it.goalFragmentInfo.obfuscatedName == sessionFragmentInfo.fragmentName) &&
                        it.goalFragmentInfo.fragmentId == sessionFragmentInfo.fragmentId
            }.map { it as FragmentReachGoalData }
            .toList()
    }

    fun getButtonClickGoals(activityName: String): List<ButtonClickGoalData> {
        return definedGoalsDataSet.keys()
            .asSequence()
            .filter {
                it is ButtonClickGoalData &&
                        it.activityClassName == activityName &&
                        it.goalFragmentInfo == null
            }.map { it as ButtonClickGoalData }
            .toList()
    }

    fun getButtonClickGoals(sessionFragmentInfo: SessionFragmentInfo): List<ButtonClickGoalData> {
        return definedGoalsDataSet.keys()
            .asSequence()
            .filter {
                it is ButtonClickGoalData &&
                        it.goalFragmentInfo != null &&
                        it.activityClassName == sessionFragmentInfo.activityName &&
                        it.goalFragmentInfo.fragmentId == sessionFragmentInfo.fragmentId &&
                        (it.goalFragmentInfo.actualName == sessionFragmentInfo.fragmentName ||
                                it.goalFragmentInfo.obfuscatedName == sessionFragmentInfo.fragmentName)
            }.map { it as ButtonClickGoalData }
            .toList()
    }

    fun viewGoalsByActivity(activityName: String): List<ViewGoalData> {
        return definedViewGoalsDataSet.keys.filter {
            it.goalFragmentInfo == null &&
                    it.activityClassName == activityName
        }
    }

    fun viewGoalsByFragment(sessionFragmentInfo: SessionFragmentInfo): List<ViewGoalData> {
        return definedViewGoalsDataSet.keys.filter {
            it.goalFragmentInfo != null &&
                    it.goalFragmentInfo.fragmentId == sessionFragmentInfo.fragmentId &&
                    (it.goalFragmentInfo.actualName == sessionFragmentInfo.fragmentName ||
                            it.goalFragmentInfo.obfuscatedName == sessionFragmentInfo.fragmentName)
        }
    }

    /**
     * Extracts the view of each [ViewGoalData] in the list given from the activity and updates
     * their [ViewGoalData.currentValue] with the value of the view
     */
    fun updateViewGoalValues(viewGoalDataSet: List<ViewGoalData>, activity: Activity) {
        var view: View?
        for (viewGoalData in viewGoalDataSet) {
            view = ViewExtractor.extractView(viewGoalData, activity)
            if (view != null) {
                updateViewGoalValue(view, viewGoalData)
            }
        }
    }

    /**
     * Extracts the view of each [ViewGoalData] in the list given from the fragment and updates
     * their [ViewGoalData.currentValue] with the value of the view
     */
    fun updateViewGoalValues(viewGoalDataSet: List<ViewGoalData>, fragment: Fragment) {
        var view: View?
        for (viewGoalData in viewGoalDataSet) {
            view = ViewExtractor.extractView(viewGoalData, fragment)
            if (view != null) {
                updateViewGoalValue(view, viewGoalData)
            }
        }
    }

    /**
     * updates [ViewGoalData.currentValue] of the given viewGoalData with the value of the view
     *
     * errors if the type of the [ViewGoalData] does not match the type of the view extracted
     */
    private fun updateViewGoalValue(view: View, viewGoalData: ViewGoalData) {
        var typeMisMatch = false
        when (viewGoalData.viewType) {
            ViewGoalType.TEXT_VIEW -> {
                if (view is TextView) {
                    viewGoalData.currentValue = view.text.toString()
                }else {
                    typeMisMatch = true
                }
            }
            ViewGoalType.SWITCH -> {
                if (view is Switch) {
                    viewGoalData.currentValue = view.isChecked.toString()
                } else {
                    typeMisMatch = true
                }
            }
            ViewGoalType.BUTTON -> {
                if (view is Button) {
                    viewGoalData.currentValue = view.text.toString()
                } else {
                    typeMisMatch = true
                }
            }
        }
        if (typeMisMatch) {
            Plog[T_ANALYTICS, T_ANALYTICS_GOAL].error(
                "trying to update viewGoalData current value, viewGoalData type is incorrect. " +
                        "The viewGoalData will be ignored") {
                "goalName" to viewGoalData.parentGoalName
                "viewId" to viewGoalData.viewID
                "ExpectedType" to viewGoalData.viewType
                "actualType" to view.javaClass.simpleName
            }
            viewGoalData.currentValue = Constants.ANALYTICS_ERROR_VIEW_GOAL
        }
    }

    companion object {
        const val GOAL_STORE_NAME = "goal_store"
        const val DEFINED_GOALS = "defined_goals"
    }
}


