package co.britehealth.android.service;

import android.app.IntentService;
import android.content.Intent;
import android.support.v4.content.LocalBroadcastManager;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;

import co.britehealth.android.BriteHealthSdk;
import co.britehealth.android.model.Score;
import co.britehealth.android.util.DatabaseUtils;
import co.britehealth.android.util.NetworkUtils;


public class ScoreService extends IntentService {

    private static final String TAG = ScoreService.class.getSimpleName();

    public static final String ACTION = "score_action";
    public static final String SUCCESSFUL = "successful";

    private Gson mGson;
    private OkHttpClient mHttpClient;

    public ScoreService() {

        super(TAG);

        mGson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").create();
        mHttpClient = NetworkUtils.getHttpClient();

    }

    @Override
    protected void onHandleIntent(Intent intent) {

        String apiKey = intent.getStringExtra(BriteHealthSdk.EXTRA_API_KEY);
        String secret = intent.getStringExtra(BriteHealthSdk.EXTRA_DATA_SECRET);
        int start = intent.getIntExtra(BriteHealthSdk.EXTRA_DATA_START, 0);
        int limit = intent.getIntExtra(BriteHealthSdk.EXTRA_DATA_LIMIT, 1000);
        long minTimeStamp = intent.getLongExtra(BriteHealthSdk.EXTRA_DATA_MIN, 0);
        minTimeStamp = minTimeStamp / 1000;
        long maxTimeStamp = intent.getLongExtra(BriteHealthSdk.EXTRA_DATA_MAX, System.currentTimeMillis());
        maxTimeStamp = maxTimeStamp / 1000;

        int userId = DatabaseUtils.getCurrentUserId(this);

        String endpointUrl = NetworkUtils.ENDPOINT_BASE + "/assessment/brainScore?" +
                "start=" + start +
                "&limit=" + limit +
                "&start_timestamp=" + minTimeStamp +
                "&end_timestamp=" + maxTimeStamp +
                "&user_id=" + userId +
                "&api_key=" + apiKey;

        String signature = NetworkUtils.generateSignature(
                apiKey + String.valueOf(maxTimeStamp)
                        + String.valueOf(limit)
                        + String.valueOf(start)
                        + String.valueOf(minTimeStamp)
                        + String.valueOf(userId),
                secret
        );

        Request request = new Request.Builder()
                .url(endpointUrl)
                .addHeader(NetworkUtils.BRITE_HEALTH_HEADER, signature)
                .build();

        try {

            Response response = mHttpClient.newCall(request).execute();
            Intent broadcastIntent = new Intent(ACTION);

            if (response.isSuccessful()) {

                Score[] scores = mGson.fromJson(response.body().string(), Score[].class);
                ArrayList<Score> scores1 = new ArrayList<>();
                Collections.addAll(scores1, scores);

                broadcastIntent.putExtra(SUCCESSFUL, true);
                broadcastIntent.putParcelableArrayListExtra(BriteHealthSdk.EXTRA_SCORES, scores1);

            } else {

                UnsuccessfulResponse unsuccessfulResponse = mGson.fromJson(
                        response.body().string(),
                        UnsuccessfulResponse.class
                );

                broadcastIntent.putExtra(SUCCESSFUL, false);
                broadcastIntent.putExtra(NetworkUtils.STATUS_CODE, unsuccessfulResponse.statusCode);
                broadcastIntent.putExtra(NetworkUtils.ERROR_MESSAGE, unsuccessfulResponse.errorMessage);

            }

            LocalBroadcastManager
                    .getInstance(this)
                    .sendBroadcast(broadcastIntent);

        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    private class UnsuccessfulResponse {

        @SerializedName("status_code")
        private int statusCode;

        @SerializedName("error_message")
        private String errorMessage;

    }

}
