package co.britehealth.android.service;

import android.app.IntentService;
import android.content.ContentValues;
import android.content.Intent;
import android.database.Cursor;
import android.support.v4.content.LocalBroadcastManager;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;

import java.io.IOException;
import java.util.Date;

import co.britehealth.android.BriteHealthSdk;
import co.britehealth.android.BriteHealthUser;
import co.britehealth.android.model.data.DBContract;
import co.britehealth.android.util.NetworkUtils;


public class RegisterUserService extends IntentService {

    private static final String TAG = RegisterUserService.class.getSimpleName();

    public static final String ACTION = "register_user_action";
    public static final String SUCCESSFUL = "successful";

    private Gson mGson;
    private OkHttpClient mHttpClient;

    public RegisterUserService() {

        super(TAG);

        mGson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").create();
        mHttpClient = NetworkUtils.getHttpClient();

    }

    @Override
    protected void onHandleIntent(Intent intent) {

        String apiKey = intent.getStringExtra(BriteHealthSdk.EXTRA_API_KEY);
        String secret = intent.getStringExtra(BriteHealthSdk.EXTRA_DATA_SECRET);
        BriteHealthUser user = intent.getParcelableExtra(BriteHealthSdk.EXTRA_BRITE_HEALTH_USER);

        String endpointUrl = NetworkUtils.ENDPOINT_BASE + "/user/?api_key=" + apiKey;
        final MediaType JSON = MediaType.parse("application/json; charset=utf-8");

        String selectionArg;
        if (user.getEmailMD5() != null) {
            selectionArg = user.getEmailMD5();
        } else if (user.getEmail() != null) {
            selectionArg = NetworkUtils.md5(user.getEmail());
        } else {
            selectionArg = "";
        }

        Cursor cursor = getContentResolver().query(
                DBContract.BriteHealthUserTable.CONTENT_URI,
                null,
                DBContract.BriteHealthUserTable.COLUMN_EMAIL_MD5 + " = ?",
                new String[]{selectionArg},
                null
        );

        Request request;

        if (cursor.moveToFirst()) {

            user.setId(cursor.getInt(cursor.getColumnIndex(DBContract.BriteHealthUserTable._ID)));

            String signature = NetworkUtils.generateSignature(apiKey + mGson.toJson(user), secret);
            RequestBody requestBody = RequestBody.create(JSON, mGson.toJson(user));

            request = new Request.Builder()
                    .url(endpointUrl)
                    .addHeader(NetworkUtils.BRITE_HEALTH_HEADER, signature)
                    .put(requestBody)
                    .build();

        } else {

            String signature;
            RequestBody requestBody;

            if(user.getAge()>=0) {

                PostBriteHealthUserWithAge postBriteHealthUser = new PostBriteHealthUserWithAge();
                postBriteHealthUser.mEmail = user.getEmail();
                postBriteHealthUser.mEmailMD5 = user.getEmailMD5();
                postBriteHealthUser.mGender = user.getGender();
                postBriteHealthUser.mDateOfBirth = user.getDateOfBirth();
                postBriteHealthUser.mAge = user.getAge();
                postBriteHealthUser.mImageUrl = user.getImageUrl();

                signature = NetworkUtils.generateSignature(apiKey + mGson.toJson(postBriteHealthUser), secret);
                requestBody = RequestBody.create(JSON, mGson.toJson(postBriteHealthUser));

            } else {

                PostBriteHealthUserWithoutAge postBriteHealthUser = new PostBriteHealthUserWithoutAge();
                postBriteHealthUser.mEmail = user.getEmail();
                postBriteHealthUser.mEmailMD5 = user.getEmailMD5();
                postBriteHealthUser.mGender = user.getGender();
                postBriteHealthUser.mDateOfBirth = user.getDateOfBirth();
                postBriteHealthUser.mImageUrl = user.getImageUrl();

                signature = NetworkUtils.generateSignature(apiKey + mGson.toJson(postBriteHealthUser), secret);
                requestBody = RequestBody.create(JSON, mGson.toJson(postBriteHealthUser));

            }

            request = new Request.Builder()
                    .url(endpointUrl)
                    .addHeader(NetworkUtils.BRITE_HEALTH_HEADER, signature)
                    .post(requestBody)
                    .build();

        }

        cursor.close();

        try {

            Response response = mHttpClient.newCall(request).execute();
            Intent broadcastIntent = new Intent(ACTION);

            if (response.isSuccessful()) {

                getContentResolver().delete(
                        DBContract.BriteHealthUserTable.CONTENT_URI,
                        null,
                        null
                );

                BriteHealthUser briteHealthUser = mGson.fromJson(response.body().string(), BriteHealthUser.class);

                ContentValues values = new ContentValues();
                values.put(DBContract.BriteHealthUserTable._ID, briteHealthUser.getId());
                values.put(DBContract.BriteHealthUserTable.COLUMN_EMAIL, briteHealthUser.getEmail());
                values.put(DBContract.BriteHealthUserTable.COLUMN_EMAIL_MD5, briteHealthUser.getEmailMD5());
                if (briteHealthUser.getGender() != null) {
                    values.put(DBContract.BriteHealthUserTable.COLUMN_GENDER, briteHealthUser.getGender().ordinal());
                } else {
                    values.put(DBContract.BriteHealthUserTable.COLUMN_GENDER, -1);
                }
                if (briteHealthUser.getDateOfBirth() != null) {
                    values.put(
                            DBContract.BriteHealthUserTable.COLUMN_DATE_OF_BIRTH,
                            briteHealthUser.getDateOfBirth().getTime()
                    );
                } else {
                    values.put(DBContract.BriteHealthUserTable.COLUMN_DATE_OF_BIRTH, -1);
                }
                values.put(DBContract.BriteHealthUserTable.COLUMN_AGE, briteHealthUser.getAge());
                values.put(DBContract.BriteHealthUserTable.COLUMN_IMAGE_URL, briteHealthUser.getImageUrl());

                getContentResolver().insert(
                        DBContract.BriteHealthUserTable.CONTENT_URI,
                        values
                );

                broadcastIntent.putExtra(SUCCESSFUL, true);
                broadcastIntent.putExtra(BriteHealthSdk.EXTRA_BRITE_HEALTH_USER, briteHealthUser);

            } else {

                UnsuccessfulResponse unsuccessfulResponse = mGson.fromJson(
                        response.body().string(),
                        UnsuccessfulResponse.class
                );

                broadcastIntent.putExtra(SUCCESSFUL, false);
                broadcastIntent.putExtra(NetworkUtils.STATUS_CODE, unsuccessfulResponse.statusCode);
                broadcastIntent.putExtra(NetworkUtils.ERROR_MESSAGE, unsuccessfulResponse.errorMessage);

            }

            LocalBroadcastManager
                    .getInstance(this)
                    .sendBroadcast(broadcastIntent);

        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    private class UnsuccessfulResponse {

        @SerializedName("status_code")
        private int statusCode;

        @SerializedName("error_message")
        private String errorMessage;

    }

    private class PostBriteHealthUserWithAge {

        @SerializedName("email")
        private String mEmail;

        @SerializedName("email_md5")
        private String mEmailMD5;

        @SerializedName("gender")
        private BriteHealthSdk.Gender mGender;

        @SerializedName("date_of_birth")
        private Date mDateOfBirth;

        @SerializedName("age")
        private int mAge;

        @SerializedName("image_url")
        private String mImageUrl;

    }

    private class PostBriteHealthUserWithoutAge {

        @SerializedName("email")
        private String mEmail;

        @SerializedName("email_md5")
        private String mEmailMD5;

        @SerializedName("gender")
        private BriteHealthSdk.Gender mGender;

        @SerializedName("date_of_birth")
        private Date mDateOfBirth;

        @SerializedName("image_url")
        private String mImageUrl;

    }

}
