package co.britehealth.android.service;

import android.app.IntentService;
import android.content.Intent;
import android.os.Bundle;
import android.support.v4.content.LocalBroadcastManager;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import java.io.IOException;
import java.util.Date;

import co.britehealth.android.BriteHealthSdk;
import co.britehealth.android.util.DatabaseUtils;
import co.britehealth.android.util.NetworkUtils;


public class NextAssessmentTimesService extends IntentService {

    private static final String TAG = NextAssessmentTimesService.class.getSimpleName();

    public static final String ACTION = "next_assessment_time_action";
    public static final String SUCCESSFUL = "successful";

    private Gson mGson;
    private OkHttpClient mHttpClient;

    public NextAssessmentTimesService() {

        super(TAG);

        mGson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").create();
        mHttpClient = NetworkUtils.getHttpClient();

    }

    @Override
    protected void onHandleIntent(Intent intent) {

        String apiKey = intent.getStringExtra(BriteHealthSdk.EXTRA_API_KEY);
        String secret = intent.getStringExtra(BriteHealthSdk.EXTRA_DATA_SECRET);

        int userId = DatabaseUtils.getCurrentUserId(this);

        String endpointUrl = NetworkUtils.ENDPOINT_BASE + "/assessment/nextDates?" +
                "user_id=" + userId +
                "&api_key=" + apiKey;

        String signature = NetworkUtils.generateSignature(apiKey + userId, secret);

        Request request = new Request.Builder()
                .url(endpointUrl)
                .addHeader(NetworkUtils.BRITE_HEALTH_HEADER, signature)
                .build();

        try {

            Response response = mHttpClient.newCall(request).execute();
            Intent broadcastIntent = new Intent(ACTION);

            if (response.isSuccessful()) {

                AssessmentDates assessmentDates = mGson.fromJson(response.body().string(), AssessmentDates.class);

                Bundle bundle = new Bundle();
                bundle.putLong(BriteHealthSdk.STROOP, assessmentDates.stroopDate.getTime());
                bundle.putLong(BriteHealthSdk.TRAIL_MAKING, assessmentDates.trailMakingDate.getTime());
                bundle.putLong(BriteHealthSdk.SHAPE_RECALL, assessmentDates.shapeRecallDate.getTime());
                bundle.putLong(BriteHealthSdk.DSST, assessmentDates.dsstDate.getTime());
                bundle.putLong(BriteHealthSdk.DISSIMILAR_SHAPES, assessmentDates.dissimilarShapesDate.getTime());

                broadcastIntent.putExtra(SUCCESSFUL, true);
                broadcastIntent.putExtra(BriteHealthSdk.EXTRA_NEXT_ASSESSMENT_TIMES, bundle);

            } else {

                UnsuccessfulResponse unsuccessfulResponse = mGson.fromJson(
                        response.body().string(),
                        UnsuccessfulResponse.class
                );

                broadcastIntent.putExtra(SUCCESSFUL, false);
                broadcastIntent.putExtra(NetworkUtils.STATUS_CODE, unsuccessfulResponse.statusCode);
                broadcastIntent.putExtra(NetworkUtils.ERROR_MESSAGE, unsuccessfulResponse.errorMessage);

            }

            LocalBroadcastManager
                    .getInstance(this)
                    .sendBroadcast(broadcastIntent);

        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    private class AssessmentDates {

        @SerializedName("Stroop")
        private Date stroopDate;

        @SerializedName("TrailMaking")
        private Date trailMakingDate;

        @SerializedName("ShapeRecall")
        private Date shapeRecallDate;

        @SerializedName("DSST")
        private Date dsstDate;

        @SerializedName("DissimilarShapes")
        private Date dissimilarShapesDate;

    }

    private class UnsuccessfulResponse {

        @SerializedName("status_code")
        private int statusCode;

        @SerializedName("error_message")
        private String errorMessage;

    }

}
