package co.britehealth.android.assessment.tmt;

import android.content.Context;
import android.content.pm.PackageManager;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.os.Vibrator;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import java.util.ArrayList;

import co.britehealth.android.R;


public class TmtView extends View implements View.OnTouchListener {

    private static final int PADDING = 50;

    private static final int DEFAULT_CIRCLE_COLOR = Color.rgb(16, 22, 27);

    private Paint mCirclePaint;
    private Paint mLabelPaint;
    private Paint mCorrectPaint;
    private Paint mWrongPaint;
    private Paint mCorrectPathPaint;
    private Paint mWrongPathPaint;
    private Path mCorrectPath;
    private Path mWrongPath;

    private String mMode = "N";

    private ArrayList<Float> mXCenters;
    private ArrayList<Float> mYCenters;
    private ArrayList<String> mLabels;
    private float mRadius;

    private int mCurrentIndex = -1;
    private int mClickedIndex = 0;
    private boolean mAnswer = true;
    private boolean vibrationPermitted;

    private OnCircleClickListener mListener;

    public TmtView(Context context) {

        super(context);
        initialize();

    }

    public TmtView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public TmtView(Context context, AttributeSet attrs, int defStyleAttr) {

        super(context, attrs, defStyleAttr);
        initialize();

    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {

        super.onSizeChanged(w, h, oldw, oldh);

        mXCenters = computeHorizontalCenters(w - PADDING);
        mYCenters = computeVerticalCenters(h - PADDING);
        mRadius = (float) computeRadius(w, h);

    }

    @Override
    protected void onDraw(Canvas canvas) {

        super.onDraw(canvas);

        mCorrectPath.reset();
        mWrongPath.reset();

        mCorrectPath.moveTo(
                PADDING + mXCenters.get(0),
                PADDING + mYCenters.get(0)
        );

        for (int i = 0; i < mCurrentIndex; i++) {

            mCorrectPath.lineTo(
                    PADDING + mXCenters.get(i),
                    PADDING + mYCenters.get(i)
            );

        }

        canvas.drawPath(mCorrectPath, mCorrectPathPaint);

        if (mAnswer) {

            if (mCurrentIndex > 0) {

                mCorrectPath.lineTo(
                        PADDING + mXCenters.get(mCurrentIndex),
                        PADDING + mYCenters.get(mCurrentIndex)
                );

                canvas.drawPath(mCorrectPath, mCorrectPathPaint);

            }

            for (int i = 0; i <= mCurrentIndex; i++) {

                canvas.drawCircle(
                        PADDING + mXCenters.get(i),
                        PADDING + mYCenters.get(i),
                        mRadius,
                        mCorrectPaint
                );

                canvas.drawText(
                        mLabels.get(i),
                        PADDING + mXCenters.get(i),
                        PADDING + 15 + mYCenters.get(i),
                        mLabelPaint
                );

            }

            for (int i = mCurrentIndex + 1; i < mXCenters.size(); i++) {

                canvas.drawCircle(
                        PADDING + mXCenters.get(i),
                        PADDING + mYCenters.get(i),
                        mRadius,
                        mCirclePaint
                );

                canvas.drawText(
                        mLabels.get(i),
                        PADDING + mXCenters.get(i),
                        PADDING + 15 + mYCenters.get(i),
                        mLabelPaint
                );

            }

        } else {

            if (mCurrentIndex > -1) {

                mCorrectPath.lineTo(
                        PADDING + mXCenters.get(mCurrentIndex),
                        PADDING + mYCenters.get(mCurrentIndex)
                );

                canvas.drawPath(mCorrectPath, mCorrectPathPaint);

                mWrongPath.moveTo(
                        PADDING + mXCenters.get(mCurrentIndex),
                        PADDING + mYCenters.get(mCurrentIndex)
                );

                mWrongPath.lineTo(
                        PADDING + mXCenters.get(mClickedIndex),
                        PADDING + mYCenters.get(mClickedIndex)
                );

                canvas.drawPath(mWrongPath, mWrongPathPaint);

            }

            for (int i = 0; i <= mCurrentIndex; i++) {

                canvas.drawCircle(
                        PADDING + mXCenters.get(i),
                        PADDING + mYCenters.get(i),
                        mRadius,
                        mCorrectPaint
                );

                canvas.drawText(
                        mLabels.get(i),
                        PADDING + mXCenters.get(i),
                        PADDING + 15 + mYCenters.get(i),
                        mLabelPaint
                );

            }

            for (int i = mCurrentIndex + 1; i < mClickedIndex; i++) {

                canvas.drawCircle(
                        PADDING + mXCenters.get(i),
                        PADDING + mYCenters.get(i),
                        mRadius,
                        mCirclePaint
                );

                canvas.drawText(
                        mLabels.get(i),
                        PADDING + mXCenters.get(i),
                        PADDING + 15 + mYCenters.get(i),
                        mLabelPaint
                );

            }

            canvas.drawCircle(
                    PADDING + mXCenters.get(mClickedIndex),
                    PADDING + mYCenters.get(mClickedIndex),
                    mRadius,
                    mWrongPaint
            );

            canvas.drawText(
                    mLabels.get(mClickedIndex),
                    PADDING + mXCenters.get(mClickedIndex),
                    PADDING + 15 + mYCenters.get(mClickedIndex),
                    mLabelPaint
            );

            for (int i = mClickedIndex + 1; i < mXCenters.size(); i++) {

                canvas.drawCircle(
                        PADDING + mXCenters.get(i),
                        PADDING + mYCenters.get(i),
                        mRadius,
                        mCirclePaint
                );

                canvas.drawText(
                        mLabels.get(i),
                        PADDING + mXCenters.get(i),
                        PADDING + 15 + mYCenters.get(i),
                        mLabelPaint
                );

            }

        }

    }

    private void initialize() {

        mCirclePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mCirclePaint.setStyle(Paint.Style.FILL);
        mCirclePaint.setColor(DEFAULT_CIRCLE_COLOR);

        mCorrectPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mCorrectPaint.setStyle(Paint.Style.FILL);
        mCorrectPaint.setColor(getResources().getColor(R.color.brite_health_stroop_green));

        mWrongPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mWrongPaint.setStyle(Paint.Style.FILL);
        mWrongPaint.setColor(getResources().getColor(R.color.brite_health_stroop_red));

        mLabelPaint = new Paint(Paint.FAKE_BOLD_TEXT_FLAG);
        mLabelPaint.setTextAlign(Paint.Align.CENTER);
        mLabelPaint.setColor(Color.WHITE);
        mLabelPaint.setTextSize(40);

        mCorrectPath = new Path();
        mWrongPath = new Path();

        mCorrectPathPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mCorrectPathPaint.setStyle(Paint.Style.STROKE);
        mCorrectPathPaint.setStrokeWidth(15);
        mCorrectPathPaint.setColor(getResources().getColor(R.color.brite_health_stroop_green));

        mWrongPathPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mWrongPathPaint.setStyle(Paint.Style.STROKE);
        mWrongPathPaint.setStrokeWidth(15);
        mWrongPathPaint.setColor(getResources().getColor(R.color.brite_health_stroop_red));

        setOnTouchListener(this);

        vibrationPermitted = isVibrationPermitted();

        mLabels = new ArrayList<>();
        if (mMode.equals("N")) {

            for (int i = 0; i < 23; i++) {
                mLabels.add(String.valueOf(i + 1));
            }

            mLabels.add("End");

        }

    }

    private double computeRadius(int width, int height) {
        return ((Math.sqrt(((width - PADDING) * (height - PADDING)) / 100))) / 2.1;
    }

    private ArrayList<Float> computeHorizontalCenters(int width) {

        ArrayList<Float> arrayList = new ArrayList<>();

        arrayList.add(width * 0.58f);
        arrayList.add(width * 0.4f);
        arrayList.add(width * 0.65f);
        arrayList.add(width * 0.65f);
        arrayList.add(width * 0.40f);
        arrayList.add(width * 0.58f);
        arrayList.add(width * 0.41f);
        arrayList.add(width * 0.2f);
        arrayList.add(width * 0.31f);
        arrayList.add(width * 0.08f);
        arrayList.add(width * 0.11f);
        arrayList.add(width * 0.08f);
        arrayList.add(width * 0.08f);
        arrayList.add(width * 0.11f);
        arrayList.add(width * 0.25f);
        arrayList.add(width * 0.33f);
        arrayList.add(width * 0.58f);
        arrayList.add(width * 0.4f);
        arrayList.add(width * 0.5f);
        arrayList.add(width * 0.77f);
        arrayList.add(width * 0.92f);
        arrayList.add(width * 0.75f);
        arrayList.add(width * 0.78f);
        arrayList.add(width * 0.82f);

        return arrayList;

    }

    private ArrayList<Float> computeVerticalCenters(int height) {

        ArrayList<Float> arrayList = new ArrayList<>();

        arrayList.add(height * 0.7f);
        arrayList.add(height * 0.85f);
        arrayList.add(height * 0.85f);
        arrayList.add(height * 0.4f);
        arrayList.add(height * 0.5f);
        arrayList.add(height * 0.57f);
        arrayList.add(height * 0.71f);
        arrayList.add(height * 0.8f);
        arrayList.add(height * 0.91f);
        arrayList.add(height * 0.91f);
        arrayList.add(height * 0.52f);
        arrayList.add(height * 0.64f);
        arrayList.add(height * 0.11f);
        arrayList.add(height * 0.28f);
        arrayList.add(height * 0.05f);
        arrayList.add(height * 0.28f);
        arrayList.add(height * 0.17f);
        arrayList.add(height * 0.15f);
        arrayList.add(height * 0.03f);
        arrayList.add(height * 0.08f);
        arrayList.add(height * 0.55f);
        arrayList.add(height * 0.25f);
        arrayList.add(height * 0.5f);
        arrayList.add(height * 0.85f);

        return arrayList;

    }

    /* View.OnTouchListener */

    @Override
    public boolean onTouch(View v, MotionEvent event) {

        if (event.getAction() == MotionEvent.ACTION_DOWN) {

            for (int i = 0; i < mXCenters.size(); i++) {

                float x = event.getX();
                float y = event.getY();
                float centerX = PADDING + mXCenters.get(i);
                float centerY = PADDING + mYCenters.get(i);

                if (x > (centerX - mRadius) && x < (centerX + mRadius)) {

                    if (y > (centerY - mRadius) && y < (centerY + mRadius)) {

                        mClickedIndex = i;

                        if ((i - 1) == mCurrentIndex) {

                            mAnswer = true;
                            mListener.onCircleClick(i, true);
                            mCurrentIndex++;

                        } else {

                            if (i > mCurrentIndex) {

                                mAnswer = false;
                                mListener.onCircleClick(i, false);

                                if (vibrationPermitted) {

                                    Vibrator vibrator;
                                    vibrator = (Vibrator) getContext().getSystemService(Context.VIBRATOR_SERVICE);
                                    vibrator.vibrate(200);

                                }

                            }

                        }

                        invalidate();
                        return true;

                    }

                }

            }

        }

        return true;

    }

    /* End of View.OnTouchListener */

    /* TmtView */

    public void setCurrentIndex(int currentIndex) {
        mCurrentIndex = currentIndex;
    }

    public void setClickedIndex(int clickedIndex) {
        mClickedIndex = clickedIndex;
    }

    public void setAnswer(boolean answer) {

        mAnswer = answer;
        invalidate();

    }

    public int getCurrentIndex() {
        return mCurrentIndex;
    }

    public int getClickedIndex() {
        return mClickedIndex;
    }

    public boolean isAnswer() {
        return mAnswer;
    }

    public void resetTmt() {

        mCurrentIndex = -1;
        mClickedIndex = 0;
        mAnswer = true;

        initialize();
        invalidate();

    }

    public void setOnCircleClickListener(OnCircleClickListener listener) {
        mListener = listener;
    }

    private boolean isVibrationPermitted() {

        String permission = "android.permission.VIBRATE";
        int res = getContext().checkCallingOrSelfPermission(permission);
        return (res == PackageManager.PERMISSION_GRANTED);

    }

    public void setMode(String mode) {

        mMode = mode;

        if (mMode.equals("N")) {

            for (int i = 0; i < 23; i++) {
                mLabels.add(String.valueOf(i + 1));
            }

            mLabels.add("End");

        } else if (mMode.equals("N&L")) {

            mLabels.clear();
            mLabels.add("1");
            mLabels.add("A");
            mLabels.add("2");
            mLabels.add("B");
            mLabels.add("3");
            mLabels.add("C");
            mLabels.add("4");
            mLabels.add("D");
            mLabels.add("5");
            mLabels.add("E");
            mLabels.add("6");
            mLabels.add("F");
            mLabels.add("7");
            mLabels.add("G");
            mLabels.add("8");
            mLabels.add("H");
            mLabels.add("9");
            mLabels.add("I");
            mLabels.add("10");
            mLabels.add("J");
            mLabels.add("11");
            mLabels.add("K");
            mLabels.add("12");
            mLabels.add("End");

        }

        invalidate();

    }

    /* End of TmtView */

    public interface OnCircleClickListener {
        void onCircleClick(int index, boolean answer);
    }

}
