package co.britehealth.android.assessment.tmt;

import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.os.Handler;
import android.os.SystemClock;
import android.support.annotation.Nullable;
import android.support.v7.widget.Toolbar;
import android.view.LayoutInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.google.android.gms.analytics.HitBuilders;
import com.google.android.gms.analytics.Tracker;

import co.britehealth.android.R;
import co.britehealth.android.assessment.BriteHealthAssessmentActivity;
import co.britehealth.android.assessment.framework.BaseTestScores;
import co.britehealth.android.assessment.framework.BaseTestView;
import co.britehealth.android.model.data.DBContract;


public class TestViewA extends BaseTestView
        implements
        TmtView.OnCircleClickListener,
        Toolbar.OnMenuItemClickListener {

    private static final String SCREEN_NAME = "Trail Making A: " + TestViewA.class.getSimpleName();
    private Tracker mTracker;

    private static final int COUNTDOWN_INITIAL_VALUE = 60; // Seconds
    private static final long NEXT_ASSESSMENT_DELAY = 2 * 7 * 24 * 60 * 60 * 1000;

    private TextView mTimerTextView;
    private TmtView mTmtView;

    private CountDownTimer mCountdownTimer;

    private long currentTick;
    private int mCurrentIndex;
    private int mClickedIndex;
    private boolean mLastAnswer;

    private int mNumErrors;

    private Handler mHandler;
    private Runnable mRunnable;
    private long mStartTime;
    private long mTimeInMilliseconds;
    private long mTimeBuffer;

    @Override
    public void onCreate(Bundle savedInstanceState) {

        super.onCreate(savedInstanceState);

        BriteHealthAssessmentActivity activity = (BriteHealthAssessmentActivity) getActivity();
        mTracker = activity.getDefaultTracker();

        setTestManager(
                new TestManager(getActivity(), this)
        );
        getTestManager().createTest();

    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {

        View rootView = inflater.inflate(R.layout.co_britehealth_android_tmt, container, false);

        Toolbar toolbar = (Toolbar) rootView.findViewById(R.id.co_britehealth_android_toolbar);
        toolbar.inflateMenu(R.menu.co_britehealth_android_test);
        toolbar.setOnMenuItemClickListener(this);

        TextView labelTextView = (TextView) rootView.findViewById(R.id.co_britehealth_android_tmt_text);
        labelTextView.setText(getString(R.string.co_britehealth_android_tmt_a));

        mTimerTextView = (TextView) toolbar.findViewById(R.id.co_britehealth_android_timer_textView);
        mCountdownTimer = new CountDownTimer(COUNTDOWN_INITIAL_VALUE * 1000, 1000) {

            long minutes;
            long seconds;

            @Override
            public void onTick(long millisUntilFinished) {

                minutes = ((millisUntilFinished + 200) / 1000) / 60;
                seconds = ((millisUntilFinished + 200) / 1000) % 60;

                mTimerTextView.setText(
                        String.format("%02d:%02d", minutes, seconds)
                );
                currentTick = millisUntilFinished;

            }

            @Override
            public void onFinish() {
                getTestManager().finishTest(System.currentTimeMillis());
            }

        };

        if (mTmtView == null) {

            mTmtView = (TmtView) rootView.findViewById(R.id.co_britehealth_android_tmtView);
            mTmtView.setMode("N");
            mTmtView.setOnCircleClickListener(this);

        } else {

            mTmtView = (TmtView) rootView.findViewById(R.id.co_britehealth_android_tmtView);
            mTmtView.setMode("N");
            mTmtView.setOnCircleClickListener(this);
            mTmtView.setCurrentIndex(mCurrentIndex);
            mTmtView.setClickedIndex(mClickedIndex);
            mTmtView.setAnswer(mLastAnswer);

        }

        return rootView;

    }

    @Override
    public void onResume() {

        super.onResume();

        mTracker.setScreenName(SCREEN_NAME);
        mTracker.send(new HitBuilders.ScreenViewBuilder().build());

        SharedPreferences preferences =
                getActivity().getSharedPreferences(TestManager.TMT_NUM_QUITS, Context.MODE_PRIVATE);

        int numQuits = preferences.getInt(TestManager.TMT_NUM_QUITS_KEY, 0);
        setRestarts(numQuits);

    }

    /* TestCallbacks */

    @Override
    public void onCreateTest() {

        ((TestManager) getTestManager()).setMode("N");
        createProgressTimer();

    }

    @Override
    public void onStartTest(long startTime) {

        mCountdownTimer.start();
        startProgressTimer();

    }

    @Override
    public void onPauseTest(long pauseTime) {

        stopProgressTimer();

        mCountdownTimer.cancel();
        mCountdownTimer = null;

        mCurrentIndex = mTmtView.getCurrentIndex();
        mClickedIndex = mTmtView.getClickedIndex();
        mLastAnswer = mTmtView.isAnswer();

    }

    @Override
    public void onResumeTest(long resumeTime) {

        startProgressTimer();

        mCountdownTimer = new CountDownTimer(currentTick, 1000) {

            @Override
            public void onTick(long millisUntilFinished) {
                mTimerTextView.setText(String.format("00:%02d", ((millisUntilFinished + 200) / 1000)));
                currentTick = millisUntilFinished;
            }

            @Override
            public void onFinish() {
                getTestManager().finishTest(System.currentTimeMillis());
            }

        };

        mCountdownTimer.start();

    }

    @Override
    public void onFinishTest(long finishTime) {

        stopProgressTimer();
        ((TestManager) getTestManager()).setProcessingTime(mTimeBuffer);

        mCountdownTimer.cancel();

        Bundle bundle = new Bundle();
        bundle.putLong("section_a_score", COUNTDOWN_INITIAL_VALUE - (currentTick / 1000));
        TestViewB testViewB = new TestViewB();
        testViewB.setArguments(bundle);

        float score = (float)
                (100f - (100f * (Math.log(COUNTDOWN_INITIAL_VALUE - (currentTick / 1000f)) / Math.log(60f))));

        getActivity()
                .getSharedPreferences(TestManager.TMT_A_SCORE, Context.MODE_PRIVATE)
                .edit()
                .putFloat(TestManager.TMT_A_SCORE, score)
                .commit();

        getFragmentManager()
                .beginTransaction()
                .replace(R.id.co_britehealth_android_container_main, testViewB)
                .commit();

    }

    @Override
    public void onRestartTest() {

        resetProgressTimer();

        mCountdownTimer = null;
        mCountdownTimer = new CountDownTimer(COUNTDOWN_INITIAL_VALUE * 1000, 1000) {

            @Override
            public void onTick(long millisUntilFinished) {
                mTimerTextView.setText(String.format("00:%02d", ((millisUntilFinished + 200) / 1000)));
                currentTick = millisUntilFinished;
            }

            @Override
            public void onFinish() {
                getTestManager().finishTest(System.currentTimeMillis());
            }

        };

        mTmtView.resetTmt();

    }

    @Override
    public void onDisplayTestScores(BaseTestScores baseTestScores) {

    }

    /* End of TestCallbacks */

    /* PauseScreenListener */

    @Override
    public void onQuitClick() {

        super.onQuitClick();

        SharedPreferences preferences =
                getActivity().getSharedPreferences(TestManager.TMT_NUM_QUITS, Context.MODE_PRIVATE);

        int numQuits = preferences.getInt(TestManager.TMT_NUM_QUITS_KEY, 0);

        if (numQuits >= 1) {

            long nextTime = System.currentTimeMillis() + NEXT_ASSESSMENT_DELAY;

            Cursor cursor = getActivity()
                    .getContentResolver()
                    .query(
                            DBContract.NextAssessmentTimesTable.CONTENT_URI,
                            null,
                            DBContract.NextAssessmentTimesTable._ID + " = ?",
                            new String[]{String.valueOf(TestModel.TYPE)},
                            null
                    );

            int score = 0;

            if (cursor.moveToFirst()) {

                score = cursor.getInt(
                        cursor.getColumnIndex(DBContract.NextAssessmentTimesTable.COLUMN_SCORE)
                );

                ContentValues values = new ContentValues();
                values.put(DBContract.NextAssessmentTimesTable._ID, TestModel.TYPE);
                values.put(DBContract.NextAssessmentTimesTable.COLUMN_SCORE, score);
                values.put(
                        DBContract.NextAssessmentTimesTable.COLUMN_NEXT_ASSESSMENT_TIME,
                        nextTime
                );

                getActivity()
                        .getContentResolver()
                        .update(
                                DBContract.NextAssessmentTimesTable.CONTENT_URI,
                                values,
                                DBContract.NextAssessmentTimesTable._ID + " = ?",
                                new String[]{String.valueOf(TestModel.TYPE)}
                        );

            } else {

                ContentValues values = new ContentValues();
                values.put(DBContract.NextAssessmentTimesTable._ID, TestModel.TYPE);
                values.put(DBContract.NextAssessmentTimesTable.COLUMN_SCORE, -1);
                values.put(
                        DBContract.NextAssessmentTimesTable.COLUMN_NEXT_ASSESSMENT_TIME,
                        nextTime
                );

                getActivity()
                        .getContentResolver()
                        .insert(
                                DBContract.NextAssessmentTimesTable.CONTENT_URI,
                                values
                        );

            }

            cursor.close();

            Introduction introduction = (Introduction) getAdapter().getFragmentList().get(1);
            introduction.showScore(score);
            introduction.showNextAssessment(nextTime);

            getViewPager().setEnabled(true);

            preferences.edit()
                    .putInt(TestManager.TMT_NUM_QUITS_KEY, 0)
                    .apply();

            getFragmentManager()
                    .beginTransaction()
                    .remove(this)
                    .commit();

        } else {

            numQuits++;
            preferences.edit()
                    .putInt(TestManager.TMT_NUM_QUITS_KEY, numQuits)
                    .apply();

            getFragmentManager()
                    .beginTransaction()
                    .remove(this)
                    .commit();

            Introduction introduction = (Introduction) getAdapter().getFragmentList().get(1);
            introduction.enableButtons();

            ((BriteHealthAssessmentActivity) getActivity()).showIntroduction(1);

        }

    }

    @Override
    public void onTutorialClick() {

        super.onTutorialClick();

        getFragmentManager()
                .beginTransaction()
                .addToBackStack(null)
                .replace(R.id.co_britehealth_android_container_main, new Tutorial())
                .commit();

    }

    @Override
    public void onRestartClick() {

        super.onRestartClick();

        SharedPreferences preferences =
                getActivity().getSharedPreferences(TestManager.TMT_NUM_QUITS, Context.MODE_PRIVATE);

        int numQuits = preferences.getInt(TestManager.TMT_NUM_QUITS_KEY, 0);
        numQuits++;

        preferences.edit()
                .putInt(TestManager.TMT_NUM_QUITS_KEY, numQuits)
                .apply();

    }

    @Override
    public void onResumeClick() {

        super.onResumeClick();
        getTestManager().resumeTest(System.currentTimeMillis());

    }

    /* End of PauseScreenListener */

    /* TmtView.OnCircleClickListener */

    @Override
    public void onCircleClick(int index, boolean answer) {

        if (answer) {

            ((TestManager) getTestManager()).setLastNumber(index + 1);

            if (index == 23) {
                getTestManager().finishTest(System.currentTimeMillis());
            }

        } else {

            mNumErrors++;
            ((TestManager) getTestManager()).setNumErrors(mNumErrors);

        }

    }

    /* End of TmtView.OnCircleClickListener */

    /* Toolbar.OnMenuItemClickListener */

    @Override
    public boolean onMenuItemClick(MenuItem item) {

        getTestManager().pauseTest(System.currentTimeMillis());
        showPauseScreen();

        return true;

    }

    /* End of Toolbar.OnMenuItemClickListener */

    private void createProgressTimer() {

        mHandler = new Handler();
        mRunnable = new Runnable() {

            @Override
            public void run() {

                mTimeInMilliseconds = SystemClock.uptimeMillis() - mStartTime;
                long updatedTime = mTimeBuffer + mTimeInMilliseconds;

                int seconds = (int) (updatedTime / 1000);
                int minutes = seconds / 60;
                seconds = seconds % 60;
                int milliseconds = (int) (updatedTime % 1000);

                mHandler.postDelayed(this, 0);

            }

        };

    }

    private void startProgressTimer() {

        mStartTime = SystemClock.uptimeMillis();
        mHandler.postDelayed(mRunnable, 0);

    }

    private void stopProgressTimer() {

        mTimeBuffer += mTimeInMilliseconds;
        mHandler.removeCallbacks(mRunnable);

    }

    private void resetProgressTimer() {

        mStartTime = 0;
        mTimeInMilliseconds = 0;
        mTimeBuffer = 0;
        mHandler.removeCallbacks(mRunnable);
        mHandler = null;
        mRunnable = null;

    }

}
