package co.britehealth.android.assessment.stroop;

import android.content.Context;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Random;

import co.britehealth.android.BriteHealthException;
import co.britehealth.android.R;
import co.britehealth.android.assessment.framework.BaseTestManager;
import co.britehealth.android.assessment.framework.BaseTestScores;
import co.britehealth.android.assessment.framework.TestCallbacks;
import co.britehealth.android.util.DatabaseUtils;


public class TestManager extends BaseTestManager {

    public static final String STROOP_NUM_QUITS = "co_britehealth_android_stroop_num_quits";
    public static final String STROOP_NUM_QUITS_KEY = "co_britehealth_android_stroop_num_quits_key";

    public static final String[] WORDS = {
            "Blue",
            "Orange",
            "Red",
            "Green",
            "Purple",
    };

    public static final int[] COLORS_ID = {
            R.color.brite_health_stroop_blue,
            R.color.brite_health_stroop_orange,
            R.color.brite_health_stroop_red,
            R.color.brite_health_stroop_green,
            R.color.brite_health_stroop_purple,
    };

    public static final int NUM_TRIALS = 20;

    private int mLocation;

    public TestManager(Context context, TestCallbacks testCallbacks) {
        super(context, testCallbacks);
        mLocation = 0;
    }

    @Override
    public TestModel getTestModel() {
        return (TestModel) super.getTestModel();
    }

    public int getLocation() {
        return mLocation;
    }

    public String getWord() {
        return getTestModel().getTestDataList().get(mLocation).getWord();
    }

    public int getColorId() {
        return COLORS_ID[getTestModel().getTestDataList().get(mLocation).getColorIndex()];
    }

    public void setTrialStartTime(long startTime) {
        getTestModel().getTestDataList().get(mLocation).setStartDate(new Date(startTime));
    }

    public void setTrialFinishTime(long finishTime) {
        getTestModel().getTestDataList().get(mLocation).setFinishDate(new Date(finishTime));
    }

    public void setProcessingTime(long processingTime) {
        getTestModel().getTestDataList().get(mLocation).setProcessingTime(Math.min(processingTime, 5000));
    }

    public boolean setUserAnswer(boolean answer) {

        boolean matching =
                getTestModel().getTestDataList().get(mLocation).getWordIndex()
                        == getTestModel().getTestDataList().get(mLocation).getColorIndex();

        if (answer == matching) {

            getTestModel().getTestDataList().get(mLocation).setCorrect(true);
            getTestModel().setNumCorrect(getTestModel().getNumCorrect() + 1);
            mLocation++;

            return true;

        } else {

            getTestModel().getTestDataList().get(mLocation).setCorrect(false);
            getTestModel().setNumWrong(getTestModel().getNumWrong() + 1);
            mLocation++;

            return false;

        }

    }

    public void timeout() {

        getTestModel().getTestDataList().get(mLocation).setCorrect(false);
        getTestModel().setNumWrong(getTestModel().getNumWrong() + 1);
        mLocation++;

    }

    @Override
    public void createTest() {

        long seed = System.nanoTime();
        Random random = new Random(seed);

        TestModel testModel = new TestModel(TestModel.TYPE);
        testModel.setNumTrials(NUM_TRIALS);

        int userId = DatabaseUtils.getCurrentUserId(getContext());
        if (userId != -1) {
            testModel.setUserId(userId);
        } else {

            throw new BriteHealthException(
                    "You need to register the current user with Brite Health before creating a test."
            );

        }

        List<TestTrial> matchingDataList = new ArrayList<>();
        List<TestTrial> mismatchedDataList = new ArrayList<>();
        List<TestTrial> dataList;

        for (int i = 0; i < TestManager.NUM_TRIALS / 2; i++) {

            matchingDataList.add(
                    new TestTrial(i % 5, i % 5)
            );

        }

        for (int i = 0; i < TestManager.NUM_TRIALS / 2; i++) {

            int firstRandomInt = random.nextInt(5);
            int secondRandomInt;

            // Todo: Although highly unlikely, this could potentially be an infinite loop. Needs to be modified.
            do {
                secondRandomInt = random.nextInt(5);
            } while (secondRandomInt == firstRandomInt);

            mismatchedDataList.add(
                    new TestTrial(
                            firstRandomInt,
                            secondRandomInt
                    ));

        }

        matchingDataList.addAll(mismatchedDataList);
        dataList = matchingDataList;
        Collections.shuffle(dataList, random);

        testModel.setTestDataList(dataList);

        setTestModel(testModel);

        super.createTest();

    }

    @Override
    public void restartTest(long restartTime) {

        mLocation = 0;
        super.restartTest(restartTime);

    }

    @Override
    public BaseTestScores computeBasicTestScores() {

        double accuracyScore = 0.25 * (getTestModel().getNumCorrect());
        long[] reactionTimes = new long[20];

        for (int i = 0; i < getTestModel().getTestDataList().size(); i++) {

            reactionTimes[i] = (getTestModel().getTestDataList().get(i).getFinishDate().getTime())
                    - (getTestModel().getTestDataList().get(i).getStartDate().getTime());

        }

        Arrays.sort(reactionTimes);
        double median;
        if (reactionTimes.length % 2 == 0) {
            median = ((double) reactionTimes[reactionTimes.length / 2] + (double) reactionTimes[reactionTimes.length / 2 - 1]) / 2;

        } else {
            median = (double) reactionTimes[reactionTimes.length / 2];
        }

        if (median > 3000) {
            median = 3000;
        } else if (median < 500) {
            median = 500;
        }

        double reactionTimeScore = 5 - 5 * (Math.log(median) - Math.log(500)) / (Math.log(3000) - Math.log(500));

        double finalScore = accuracyScore;
        if (accuracyScore >= 3.5) {
            finalScore = finalScore + reactionTimeScore;
        }

        TestScores testScores = new TestScores();
        testScores.setScore((int) (10 * finalScore));
        return testScores;

    }

}
