package co.britehealth.android.assessment.common;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;

import co.britehealth.android.R;


public class BulletProgressBar extends View {

    private static final int DEFAULT_BACKGROUND_COLOR = Color.DKGRAY;
    private static final float DEFAULT_BORDER_WIDTH = 20f;
    private static final float DEFAULT_SHADOW_RADIUS = 0f;
    private static final int DEFAULT_LENGTH = 5;
    private static final int DEFAULT_PROGRESS = 0;

    private int mBackgroundColor;
    private int mBulletColor;
    private float mBorderWidth;
    private float mShadowRadius;
    private int mLength;
    private int mProgress;

    private Paint mBackgroundPaint;
    private Paint mBulletPaint;

    public BulletProgressBar(Context context) {

        super(context);
        initialize();

    }

    public BulletProgressBar(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public BulletProgressBar(Context context, AttributeSet attrs, int defStyleAttr) {

        super(context, attrs, defStyleAttr);

        TypedArray typedArray = context.getTheme()
                .obtainStyledAttributes(
                        attrs,
                        R.styleable.BulletProgressBar,
                        defStyleAttr,
                        0
                );

        try {

            mBackgroundColor = typedArray.getColor(
                    R.styleable.BulletProgressBar_backgroundColor,
                    DEFAULT_BACKGROUND_COLOR
            );

            mBulletColor = typedArray.getColor(
                    R.styleable.BulletProgressBar_bulletColor,
                    getResources().getColor(R.color.co_britehealth_android_orange_dark)
            );

            mBorderWidth = typedArray.getDimension(
                    R.styleable.BulletProgressBar_bulletBorderWidth,
                    DEFAULT_BORDER_WIDTH
            );

            mShadowRadius = typedArray.getDimension(
                    R.styleable.BulletProgressBar_shadowRadius,
                    DEFAULT_SHADOW_RADIUS
            );

            mLength = typedArray.getInt(
                    R.styleable.BulletProgressBar_length,
                    DEFAULT_LENGTH
            );

            mProgress = typedArray.getInt(
                    R.styleable.BulletProgressBar_progress,
                    DEFAULT_PROGRESS
            );

            mProgress = Math.min(mProgress, mLength);

        } finally {

            typedArray.recycle();

        }

        initialize();

    }

    @Override
    protected void onDraw(Canvas canvas) {

        super.onDraw(canvas);

        float radius = getWidth() / ((5 * mLength) - 1);

        canvas.drawRect(
                2 * radius,
                (getHeight() / 2) - (radius / 4),
                getWidth() - (2 * radius),
                (getHeight() / 2) + (radius / 4),
                mBackgroundPaint
        );

        for (int i = 0; i < mLength; i++) {

            canvas.drawCircle(
                    radius * ((5 * i) + 2),
                    getHeight() / 2,
                    radius,
                    mBackgroundPaint
            );

        }

        canvas.drawRect(
                2 * radius,
                (getHeight() / 2) - (radius / 4),
                ((mProgress - 1) * 5 * radius) + (2 * radius),
                (getHeight() / 2) + (radius / 4),
                mBulletPaint
        );

        for (int i = 0; i < mProgress; i++) {

            canvas.drawCircle(
                    radius * ((5 * i) + 2),
                    getHeight() / 2,
                    radius,
                    mBulletPaint
            );

        }

    }

    private void initialize() {

        mBackgroundPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBackgroundPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mBackgroundPaint.setStrokeWidth(mBorderWidth);
        mBackgroundPaint.setColor(mBackgroundColor);
        mBackgroundPaint.setShadowLayer(mShadowRadius, 0, mShadowRadius, mBackgroundColor);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
            setLayerType(LAYER_TYPE_SOFTWARE, mBackgroundPaint);
        }

        mBulletPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBulletPaint.setStyle(Paint.Style.FILL);
        mBulletPaint.setColor(mBulletColor);

    }

    public int getBackgroundColor() {
        return mBackgroundColor;
    }

    public int getBulletColor() {
        return mBulletColor;
    }

    public float getBorderWidth() {
        return mBorderWidth;
    }

    public float getShadowRadius() {
        return mShadowRadius;
    }

    public int getLength() {
        return mLength;
    }

    public int getProgress() {
        return mProgress;
    }

    public void setBackgroundColor(int backgroundColor) {

        mBackgroundColor = backgroundColor;

        invalidate();
        requestLayout();

    }

    public void setBulletColor(int bulletColor) {

        mBulletColor = bulletColor;

        invalidate();
        requestLayout();

    }

    public void setBorderWidth(float borderWidth) {

        mBorderWidth = borderWidth;

        invalidate();
        requestLayout();

    }

    public void setShadowRadius(float shadowRadius) {

        mShadowRadius = shadowRadius;

        invalidate();
        requestLayout();

    }

    public void setLength(int length) {

        mLength = length;

        invalidate();
        requestLayout();

    }

    public void setProgress(int progress) {

        mProgress = progress;

        invalidate();
        requestLayout();

    }

}
