package co.britehealth.android.assessment.common;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Shader;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;

import co.britehealth.android.R;


public class BriteHealthShape extends View {

    private static final int PADDING = 20;
    private static final int STROKE_WIDTH = 6;
    private static final int OFFSET = STROKE_WIDTH / 2;

    private ShapeType mShapeType;
    private ShapeColor mShapeColor;
    private ShapePattern mShapePattern;

    private Paint mBackgroundPaint;
    private Paint mShapePaint;
    private Paint mPatternPaint;

    private boolean mChecked = false;

    public BriteHealthShape(Context context) {
        super(context);
        initialize();
    }

    public BriteHealthShape(Context context, AttributeSet attrs) {
        super(context, attrs);
        initialize();
    }

    public BriteHealthShape(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initialize();
    }

    private void initialize() {

        mBackgroundPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBackgroundPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mBackgroundPaint.setStrokeWidth(STROKE_WIDTH);
        mBackgroundPaint.setColor(Color.WHITE);
        mBackgroundPaint.setShadowLayer(5, 5, 5, Color.GRAY);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
            setLayerType(LAYER_TYPE_SOFTWARE, mBackgroundPaint);
        }

        mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPatternPaint = new Paint(Paint.ANTI_ALIAS_FLAG);

    }

    public void setProperties(ShapeType type, ShapeColor color, ShapePattern pattern) {

        mShapeType = type;
        mShapeColor = color;
        mShapePattern = pattern;

        invalidate();
        requestLayout();

    }

    public boolean isChecked() {
        return mChecked;
    }

    public void setChecked(boolean checked) {

        mChecked = checked;

        initialize();
        invalidate();
        requestLayout();

    }

    @Override
    protected void onDraw(Canvas canvas) {

        switch (mShapeType) {

            case SQUARE:

                drawSquare(canvas, mShapeColor, mShapePattern);
                break;

            case CIRCLE:

                drawCircle(canvas, mShapeColor, mShapePattern);
                break;

            case S:

                drawS(canvas, mShapeColor, mShapePattern);
                break;

            case PARALLELOGRAM:

                drawParallelogram(canvas, mShapeColor, mShapePattern);
                break;

            case DIAMOND:

                drawDiamond(canvas, mShapeColor, mShapePattern);
                break;

        }

    }

    private void drawSquare(Canvas canvas, ShapeColor color, ShapePattern pattern) {

        mShapePaint = setPaintColor(mShapePaint, color);

        switch (pattern) {

            case DOTTED:

                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mShapePaint
                );

                drawDots(canvas, color);

                break;

            case STRIPES:

                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mShapePaint
                );

                drawStripes(canvas, color);

                break;

            case EMPTY:

                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mShapePaint
                );

                break;

            case SOLID:

                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.FILL_AND_STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawRect(
                        PADDING,
                        PADDING,
                        getWidth() - PADDING,
                        getHeight() - PADDING,
                        mShapePaint
                );

                break;

        }

    }

    private void drawCircle(Canvas canvas, ShapeColor color, ShapePattern pattern) {

        mShapePaint = setPaintColor(mShapePaint, color);

        float radius = ((Math.min(getWidth(), getHeight())) - (2 * PADDING)) / 2;

        Bitmap original;
        Canvas originalCanvas;
        Shader shader;

        switch (pattern) {

            case DOTTED:

                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mBackgroundPaint
                );

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawDots(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                break;

            case STRIPES:

                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mBackgroundPaint
                );

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawStripes(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                break;

            case EMPTY:

                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                break;

            case SOLID:

                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mBackgroundPaint
                );

                mShapePaint.setStyle(Paint.Style.FILL_AND_STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawCircle(
                        getWidth() / 2,
                        getHeight() / 2,
                        radius,
                        mShapePaint
                );

                break;

        }

    }

    private void drawS(Canvas canvas, ShapeColor color, ShapePattern pattern) {

        mShapePaint = setPaintColor(mShapePaint, color);

        Bitmap original;
        Canvas originalCanvas;
        Shader shader;

        int width = getWidth() - (2 * PADDING);
        int height = getHeight() - (2 * PADDING);

        Path path = new Path();
        path.moveTo(PADDING, PADDING);
        path.cubicTo(
                PADDING + (width / 3), PADDING + (2f * height),
                PADDING + ((2 * width) / 3), PADDING + (-0.5f * height),
                PADDING + width, PADDING + height
        );

        path.cubicTo(
                PADDING + ((2 * width) / 3), PADDING + (-0.9f * height),
                PADDING + (width / 3), PADDING + (1.6f * height),
                PADDING, PADDING
        );

        switch (pattern) {

            case DOTTED:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawDots(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case STRIPES:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawStripes(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case EMPTY:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

            case SOLID:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.FILL_AND_STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

        }

    }

    private void drawParallelogram(Canvas canvas, ShapeColor color, ShapePattern pattern) {

        mShapePaint = setPaintColor(mShapePaint, color);

        Bitmap original;
        Canvas originalCanvas;
        Shader shader;

        Path path = new Path();
        path.moveTo(PADDING - OFFSET, PADDING);
        path.lineTo(getWidth() / 2, PADDING);
        path.lineTo(getWidth() - PADDING, getHeight() - PADDING);
        path.lineTo(getWidth() / 2, getHeight() - PADDING);
        path.lineTo(PADDING, PADDING);

        switch (pattern) {

            case DOTTED:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawDots(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case STRIPES:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawStripes(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case EMPTY:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

            case SOLID:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.FILL_AND_STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

        }

    }

    private void drawDiamond(Canvas canvas, ShapeColor color, ShapePattern pattern) {

        mShapePaint = setPaintColor(mShapePaint, color);

        Bitmap original;
        Canvas originalCanvas;
        Shader shader;

        Path path = new Path();
        path.moveTo(getWidth() / 2, PADDING + OFFSET);
        path.lineTo(getWidth() - PADDING - OFFSET, getHeight() / 2);
        path.lineTo(getWidth() / 2, getHeight() - PADDING - OFFSET);
        path.lineTo(PADDING + OFFSET, getHeight() / 2);
        path.lineTo(getWidth() / 2, PADDING + OFFSET);

        switch (pattern) {

            case DOTTED:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawDots(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case STRIPES:

                canvas.drawPath(path, mBackgroundPaint);

                original = Bitmap.createBitmap(
                        getWidth(),
                        getHeight(),
                        Bitmap.Config.ARGB_8888
                );
                originalCanvas = new Canvas(original);
                drawStripes(originalCanvas, color);

                shader = new BitmapShader(original, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                mShapePaint.setShader(shader);
                canvas.drawPath(path, mShapePaint);

                mShapePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                mShapePaint = setPaintColor(mShapePaint, color);
                canvas.drawPath(path, mShapePaint);

                break;

            case EMPTY:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

            case SOLID:

                canvas.drawPath(path, mBackgroundPaint);

                mShapePaint.setStyle(Paint.Style.FILL_AND_STROKE);
                mShapePaint.setStrokeWidth(STROKE_WIDTH);
                canvas.drawPath(path, mShapePaint);

                break;

        }

    }

    private Paint setPaintColor(Paint paint, ShapeColor color) {

        switch (color) {

            case BLUE:

                paint.setColor(getResources().getColor(R.color.brite_health_stroop_blue));
                break;

            case ORANGE:

                paint.setColor(getResources().getColor(R.color.brite_health_stroop_orange));
                break;

            case RED:

                paint.setColor(getResources().getColor(R.color.brite_health_stroop_red));
                break;

            case GREEN:

                paint.setColor(getResources().getColor(R.color.brite_health_stroop_green));
                break;

            case PURPLE:

                paint.setColor(getResources().getColor(R.color.brite_health_stroop_purple));
                break;

        }

        return paint;

    }

    private void drawDots(Canvas canvas, ShapeColor color) {

        mPatternPaint.setStyle(Paint.Style.FILL);
        mPatternPaint = setPaintColor(mPatternPaint, color);

        float hCellWidth = (getWidth() - (2 * PADDING)) / 6;
        float vCellWidth = (getHeight() - (2 * PADDING)) / 6;

        for (int i = 1; i < 6; i++) {

            for (int j = 1; j < 6; j++) {

                canvas.drawCircle(
                        PADDING + OFFSET + (i * hCellWidth),
                        PADDING + OFFSET + (j * vCellWidth),
                        STROKE_WIDTH,
                        mPatternPaint
                );

            }

        }

    }

    private void drawStripes(Canvas canvas, ShapeColor color) {

        mPatternPaint.setStyle(Paint.Style.STROKE);
        mPatternPaint.setStrokeWidth(STROKE_WIDTH);
        mPatternPaint = setPaintColor(mPatternPaint, color);

        Path path = new Path();

        float hCellWidth = (getWidth() - (2 * PADDING)) / 6;
        float vCellWidth = (getHeight() - (2 * PADDING)) / 6;

        for (int i = 1; i < 7; i++) {

            path.moveTo(PADDING + (i * hCellWidth), PADDING);
            path.lineTo(PADDING, PADDING + (i * vCellWidth));

        }

        for (int i = 1; i < 6; i++) {

            path.moveTo(getWidth() - PADDING, PADDING - OFFSET + (i * vCellWidth));
            path.lineTo(PADDING - OFFSET + (i * hCellWidth), getHeight() - PADDING);

        }

        canvas.drawPath(path, mPatternPaint);

    }

    public enum ShapeType {

        SQUARE, CIRCLE, S, PARALLELOGRAM, DIAMOND;

        public static ShapeType getRandom() {
            return values()[(int) (Math.random() * values().length)];
        }

    }

    public enum ShapeColor {

        BLUE, ORANGE, RED, GREEN, PURPLE;

        public static ShapeColor getRandom() {
            return values()[(int) (Math.random() * values().length)];
        }

    }

    public enum ShapePattern {

        DOTTED, STRIPES, EMPTY, SOLID;

        public static ShapePattern getRandom() {
            return values()[(int) (Math.random() * values().length)];
        }

    }

}
