package co.britehealth.android;

import android.os.Parcel;
import android.os.Parcelable;

import com.google.gson.annotations.SerializedName;

import java.util.Date;


/**
 * A data class representing a Brite Health user.
 * A Brite Health user can consist of an email address, MD5 encrypted email address,
 * and other information such as gender, date of birth, age and profile image url.
 * <p/>
 * All {@link BriteHealthUser} objects must have at least a valid email address or MD5 encrypted email address.
 * All other fields are optional.
 *
 * @see BriteHealthSdk#registerUser(BriteHealthUser, BriteHealthSdk.RegisterUserCallbacks)
 */
public class BriteHealthUser implements Parcelable {

    @SerializedName("id")
    private int mId;

    @SerializedName("email")
    private String mEmail;

    @SerializedName("email_md5")
    private String mEmailMD5;

    @SerializedName("gender")
    private BriteHealthSdk.Gender mGender;

    @SerializedName("date_of_birth")
    private Date mDateOfBirth;

    @SerializedName("age")
    private int mAge;

    @SerializedName("image_url")
    private String mImageUrl;

    public BriteHealthUser(Builder builder) {

        mId = -1;
        mEmail = builder.mEmail;
        mEmailMD5 = builder.mEmailMD5;
        mGender = builder.mGender;
        mDateOfBirth = builder.mDateOfBirth;
        mAge = builder.mAge;
        mImageUrl = builder.mImageUrl;

    }

    public void setId(int id) {
        mId = id;
    }

    public int getId() {
        return mId;
    }

    public String getEmail() {
        return mEmail;
    }

    public String getEmailMD5() {
        return mEmailMD5;
    }

    public BriteHealthSdk.Gender getGender() {
        return mGender;
    }

    public Date getDateOfBirth() {
        return mDateOfBirth;
    }

    public int getAge() {
        return mAge;
    }

    public String getImageUrl() {
        return mImageUrl;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {

        dest.writeInt(mId);
        dest.writeString(mEmail);
        dest.writeString(mEmailMD5);
        dest.writeSerializable(mGender);

        if (mDateOfBirth != null) {
            dest.writeLong(mDateOfBirth.getTime());
        } else {
            dest.writeLong(-1);
        }

        dest.writeInt(mAge);
        dest.writeString(mImageUrl);

    }

    public static final Creator<BriteHealthUser> CREATOR = new Creator<BriteHealthUser>() {

        @Override
        public BriteHealthUser createFromParcel(Parcel source) {
            return new BriteHealthUser(source);
        }

        @Override
        public BriteHealthUser[] newArray(int size) {
            return new BriteHealthUser[size];
        }

    };

    private BriteHealthUser(Parcel parcel) {

        mId = parcel.readInt();
        mEmail = parcel.readString();
        mEmailMD5 = parcel.readString();
        mGender = (BriteHealthSdk.Gender) parcel.readSerializable();

        long time = parcel.readLong();
        if (time == -1) {
            mDateOfBirth = null;
        } else {
            mDateOfBirth = new Date(time);
        }

        mAge = parcel.readInt();
        mImageUrl = parcel.readString();

    }

    /**
     * Builder class for {@link BriteHealthUser} objects.
     * Provides a convenient way to set the various fields of a {@link BriteHealthUser}.
     */
    public static final class Builder {

        private String mEmail;
        private String mEmailMD5;
        private BriteHealthSdk.Gender mGender;
        private Date mDateOfBirth;
        private int mAge;
        private String mImageUrl;

        public Builder() {

            mEmail = null;
            mEmailMD5 = null;
            mGender = null;
            mDateOfBirth = null;
            mAge = -1;
            mImageUrl = null;

        }

        public Builder setEmail(String email) {
            mEmail = email;
            return this;
        }

        public Builder setEmailMD5(String emailMD5) {
            mEmailMD5 = emailMD5;
            return this;
        }

        public Builder setGender(BriteHealthSdk.Gender gender) {
            mGender = gender;
            return this;
        }

        public Builder setDateOfBirth(Date dateOfBirth) {
            mDateOfBirth = dateOfBirth;
            return this;
        }

        public Builder setAge(int age) {
            mAge = age;
            return this;
        }

        public Builder setImageUrl(String imageUrl) {
            mImageUrl = imageUrl;
            return this;
        }

        public BriteHealthUser build() {
            return new BriteHealthUser(this);
        }

    }

}
