package co.britehealth.android;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Fragment;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.content.LocalBroadcastManager;

import java.util.ArrayList;

import co.britehealth.android.analytics.ComparativeAnalyticsFragment;
import co.britehealth.android.analytics.IndividualAnalyticsFragment;
import co.britehealth.android.analytics.SupportComparativeAnalyticsFragment;
import co.britehealth.android.analytics.SupportIndividualAnalyticsFragment;
import co.britehealth.android.assessment.BriteHealthAssessmentActivity;
import co.britehealth.android.model.Score;
import co.britehealth.android.service.NextAssessmentTimesService;
import co.britehealth.android.service.RegisterUserService;
import co.britehealth.android.service.ScoreService;
import co.britehealth.android.service.UserFitnessActivityService;
import co.britehealth.android.util.NetworkUtils;


/**
 * The main entry point for Brite Health SDK integration.<p>
 * Before any operation is executed, the {@link BriteHealthSdk} must be built using {@link BriteHealthSdk.Builder}.
 */
public class BriteHealthSdk implements BriteHealthSdkInterface {

    public static final String EXTRA_API_KEY = "brite_health_api_key";
    public static final String EXTRA_DATA_SECRET = "brite_health_secret";
    public static final String EXTRA_BRITE_HEALTH_USER = "extra_brite_health_user";
    public static final String EXTRA_USER_FITNESS_ACTIVITY = "extra_user_fitness_activity";
    public static final String EXTRA_NEXT_ASSESSMENT_TIMES = "extra_next_assessment_times";
    public static final String EXTRA_DATA_START = "extra_data_start";
    public static final String EXTRA_DATA_LIMIT = "extra_data_limit";
    public static final String EXTRA_DATA_MIN = "extra_data_min";
    public static final String EXTRA_DATA_MAX = "extra_data_max";
    public static final String EXTRA_SCORES = "extra_scores";

    public static final String STROOP = "Stroop";
    public static final String TRAIL_MAKING = "TrailMaking";
    public static final String SHAPE_RECALL = "ShapeRecall";
    public static final String DSST = "DSST";
    public static final String DISSIMILAR_SHAPES = "DissimilarShapes";

    private Context mContext;
    private String mApiKey;
    private String mSecret;

    private BroadcastReceiver mRegisterUserReceiver;
    private RegisterUserCallbacks mRegisterUserCallbacks;

    private BroadcastReceiver mUserFitnessActivityReceiver;
    private UserFitnessActivityCallbacks mUserFitnessActivityCallbacks;

    private BroadcastReceiver mNextAssessmentTimesReceiver;
    private NextAssessmentTimesCallbacks mNextAssessmentTimesCallbacks;

    private BroadcastReceiver mScoreReceiver;
    private ScoreCallbacks mScoreCallbacks;

    private BriteHealthSdk(Builder builder) {

        mContext = builder.mContext;
        mApiKey = builder.mApiKey;
        mSecret = builder.mSecret;

    }

    /**
     * Registers a {@link BriteHealthUser} with Brite Health. <p>
     * This method returns immediately, and connects to the server in the background.
     * If the registration is successful, {@link RegisterUserCallbacks#onSuccess(BriteHealthUser)} is called.
     * On a failure, {@link RegisterUserCallbacks#onFailure(BriteHealthException)} is called.
     *
     * @param briteHealthUser The {@link BriteHealthUser} object containing user's information.
     * @param callbacks       An implementation of {@link RegisterUserCallbacks}.
     * @see BriteHealthUser
     */
    @Override
    public void registerUser(BriteHealthUser briteHealthUser, RegisterUserCallbacks callbacks) {

        mRegisterUserCallbacks = callbacks;

        mRegisterUserReceiver = new RegisterUserReceiver();
        LocalBroadcastManager
                .getInstance(mContext)
                .registerReceiver(
                        mRegisterUserReceiver,
                        new IntentFilter(RegisterUserService.ACTION)
                );

        Intent intent = new Intent(mContext, RegisterUserService.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);
        intent.putExtra(EXTRA_BRITE_HEALTH_USER, briteHealthUser);
        mContext.startService(intent);

    }

    /**
     * Adds new {@link FitnessActivity} for the user.<p>
     * This method returns immediately, and connects to the server in the background.
     * If the call is successful, {@link UserFitnessActivityCallbacks#onSuccess()} is called.
     * On a failure, {@link UserFitnessActivityCallbacks#onFailure(BriteHealthException)} is called.
     *
     * @param fitnessActivity The {@link FitnessActivity} object containing user's fitness activity information.
     * @param callbacks       An implementation of {@link UserFitnessActivityCallbacks}.
     * @see FitnessActivity
     */
    @Override
    public void addUserFitnessActivity(FitnessActivity fitnessActivity, UserFitnessActivityCallbacks callbacks) {

        mUserFitnessActivityCallbacks = callbacks;

        mUserFitnessActivityReceiver = new UserFitnessActivityReceiver();
        LocalBroadcastManager
                .getInstance(mContext)
                .registerReceiver(
                        mUserFitnessActivityReceiver,
                        new IntentFilter(UserFitnessActivityService.ACTION)
                );

        ArrayList<FitnessActivity> fitnessActivities = new ArrayList<>();
        fitnessActivities.add(fitnessActivity);

        Intent intent = new Intent(mContext, UserFitnessActivityService.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);
        intent.putParcelableArrayListExtra(EXTRA_USER_FITNESS_ACTIVITY, fitnessActivities);
        mContext.startService(intent);

    }

    /**
     * Adds a list of new {@link FitnessActivity} for the user.<p>
     * This method returns immediately, and connects to the server in the background.
     * If the call is successful, {@link UserFitnessActivityCallbacks#onSuccess()} is called.
     * On a failure, {@link UserFitnessActivityCallbacks#onFailure(BriteHealthException)} is called.
     *
     * @param fitnessActivities The list of {@link FitnessActivity} object containing user's fitness activity information.
     * @param callbacks         An implementation of {@link UserFitnessActivityCallbacks}.
     * @see FitnessActivity
     */
    @Override
    public void addUserFitnessActivity(ArrayList<FitnessActivity> fitnessActivities, UserFitnessActivityCallbacks callbacks) {

        mUserFitnessActivityCallbacks = callbacks;

        mUserFitnessActivityReceiver = new UserFitnessActivityReceiver();
        LocalBroadcastManager
                .getInstance(mContext)
                .registerReceiver(
                        mUserFitnessActivityReceiver,
                        new IntentFilter(UserFitnessActivityService.ACTION)
                );

        Intent intent = new Intent(mContext, UserFitnessActivityService.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);
        intent.putParcelableArrayListExtra(EXTRA_USER_FITNESS_ACTIVITY, fitnessActivities);
        mContext.startService(intent);

    }

    /**
     * Returns the next time user needs to take the assessment. <p>
     * This method returns immediately, and connects to the server in the background.
     * If the call is successful, {@link NextAssessmentTimesCallbacks#onSuccess(Bundle)} is called.
     * On a failure, {@link NextAssessmentTimesCallbacks#onFailure(BriteHealthException)} is called.
     *
     * @param callbacks An implementation of {@link NextAssessmentTimesCallbacks}.
     */
    @Override
    public void getNextAssessmentTimes(NextAssessmentTimesCallbacks callbacks) {

        mNextAssessmentTimesCallbacks = callbacks;

        mNextAssessmentTimesReceiver = new NextAssessmentTimesReceiver();
        LocalBroadcastManager
                .getInstance(mContext)
                .registerReceiver(
                        mNextAssessmentTimesReceiver,
                        new IntentFilter(NextAssessmentTimesService.ACTION)
                );

        Intent intent = new Intent(mContext, NextAssessmentTimesService.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);
        mContext.startService(intent);

    }

    /**
     * Gets the total brain health score history for the user.<p>
     * This method returns immediately, and connects to the server in the background.
     * If the call is successful, {@link ScoreCallbacks#onSuccess(ArrayList)} is called.
     * On a failure, {@link ScoreCallbacks#onFailure(BriteHealthException)} is called.
     *
     * @param start        The start for paginating results.
     * @param limit        The number of previous scores to retrieve.
     * @param minTimestamp Start time for your query. If there is no data in the specified date range,
     *                     the last available score is returned.
     * @param maxTimestamp End time for your query. If there is no data in the specified date range,
     *                     the last available score is returned.
     * @param callbacks    An implementation of {@link ScoreCallbacks}.
     */
    @Override
    public void getScore(int start, int limit, long minTimestamp, long maxTimestamp, ScoreCallbacks callbacks) {

        mScoreCallbacks = callbacks;

        mScoreReceiver = new ScoreReceiver();
        LocalBroadcastManager
                .getInstance(mContext)
                .registerReceiver(
                        mScoreReceiver,
                        new IntentFilter(ScoreService.ACTION)
                );

        Intent intent = new Intent(mContext, ScoreService.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);
        intent.putExtra(EXTRA_DATA_START, start);
        intent.putExtra(EXTRA_DATA_LIMIT, limit);
        intent.putExtra(EXTRA_DATA_MIN, minTimestamp);
        intent.putExtra(EXTRA_DATA_MAX, maxTimestamp);
        mContext.startService(intent);

    }

    /**
     * Launches the Brite Health assessment from an Activity. <p>
     * Call this method when you are ready to launch the assessment.
     * It is strongly recommended to separately request the next assessment time for the user using the
     * {@link BriteHealthSdk#getNextAssessmentTimes(NextAssessmentTimesCallbacks)} method and only call
     * this method if the assessment is due.
     *
     * @param activity    The Activity that launches the assessment.
     * @param requestCode The request code so that you can identify the request when the result is returned.
     */
    @Override
    public void launchAssessmentForResult(Activity activity, int requestCode) {

        Intent intent = new Intent(mContext, BriteHealthAssessmentActivity.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);

        activity.startActivityForResult(intent, requestCode);

    }

    /**
     * Launches the Brite Health assessment from a Fragment. <p>
     * Call this method when you are ready to launch the assessment.
     * It is strongly recommended to separately request the next assessment time for the user using the
     * {@link BriteHealthSdk#getNextAssessmentTimes(NextAssessmentTimesCallbacks)} method and only call
     * this method if the assessment is due.
     *
     * @param fragment    The Fragment that launches the assessment.
     * @param requestCode The request code so that you can identify the request when the result is returned.
     */
    @Override
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    public void launchAssessmentForResult(Fragment fragment, int requestCode) {

        Intent intent = new Intent(mContext, BriteHealthAssessmentActivity.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);

        fragment.startActivityForResult(intent, requestCode);

    }

    /**
     * Launches the Brite Health assessment from an android.support.v4.app.Fragment. <p>
     * Call this method when you are ready to launch the assessment.
     * It is strongly recommended to separately request the next assessment time for the user using the
     * {@link BriteHealthSdk#getNextAssessmentTimes(NextAssessmentTimesCallbacks)} method and only call
     * this method if the assessment is due.
     *
     * @param supportFragment The Fragment that launches the assessment.
     * @param requestCode     The request code so that you can identify the request when the result is returned.
     */
    @Override
    public void launchAssessmentForResult(android.support.v4.app.Fragment supportFragment, int requestCode) {

        Intent intent = new Intent(mContext, BriteHealthAssessmentActivity.class);
        intent.putExtra(EXTRA_API_KEY, mApiKey);
        intent.putExtra(EXTRA_DATA_SECRET, mSecret);

        supportFragment.startActivityForResult(intent, requestCode);

    }

    /**
     * Returns the Fragment containing rich and out-of-the box brain health analytics to your users.
     * You can add add this Fragment wherever you see fit in your app.
     *
     * @param analyticsType The type of the Analytics Fragment.
     * @return The Fragment containing rich and out-of-the box brain health analytics to your users.
     * @see #getSupportAnalyticsFragment(AnalyticsType)
     */
    @Override
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    public Fragment getAnalyticsFragment(AnalyticsType analyticsType) {

        Bundle bundle = new Bundle();
        bundle.putString(EXTRA_API_KEY, mApiKey);
        bundle.putString(EXTRA_DATA_SECRET, mSecret);

        switch (analyticsType) {

            case INDIVIDUAL:

                IndividualAnalyticsFragment individualAnalyticsFragment =
                        new IndividualAnalyticsFragment();
                individualAnalyticsFragment.setArguments(bundle);

                return individualAnalyticsFragment;

            case COMPARATIVE:

                ComparativeAnalyticsFragment comparativeAnalyticsFragment =
                        new ComparativeAnalyticsFragment();
                comparativeAnalyticsFragment.setArguments(bundle);

                return comparativeAnalyticsFragment;

            default:
                throw new UnsupportedOperationException("No such Analytics Fragment.");

        }

    }

    /**
     * Returns the android.support.v4.app.Fragment containing rich and out-of-the box brain health analytics to your users.
     * You can add add this Fragment wherever you see fit in your app.
     *
     * @param analyticsType The type of the Analytics Fragment.
     * @return The Fragment containing rich and out-of-the box brain health analytics to your users.
     * @see #getAnalyticsFragment(AnalyticsType)
     */
    @Override
    public android.support.v4.app.Fragment getSupportAnalyticsFragment(AnalyticsType analyticsType) {

        Bundle bundle = new Bundle();
        bundle.putString(EXTRA_API_KEY, mApiKey);
        bundle.putString(EXTRA_DATA_SECRET, mSecret);

        switch (analyticsType) {
            case INDIVIDUAL:

                SupportIndividualAnalyticsFragment supportIndividualAnalyticsFragment =
                        new SupportIndividualAnalyticsFragment();
                supportIndividualAnalyticsFragment.setArguments(bundle);

                return supportIndividualAnalyticsFragment;

            case COMPARATIVE:

                SupportComparativeAnalyticsFragment supportComparativeAnalyticsFragment =
                        new SupportComparativeAnalyticsFragment();
                supportComparativeAnalyticsFragment.setArguments(bundle);

                return supportComparativeAnalyticsFragment;

            default:
                throw new UnsupportedOperationException("No such Analytics Fragment.");
        }

    }

    private final class RegisterUserReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {

            boolean successful = intent.getBooleanExtra(RegisterUserService.SUCCESSFUL, false);

            if (successful) {

                BriteHealthUser briteHealthUser = intent.getParcelableExtra(EXTRA_BRITE_HEALTH_USER);
                mRegisterUserCallbacks.onSuccess(briteHealthUser);

            } else {

                mRegisterUserCallbacks.onFailure(
                        new BriteHealthException(
                                intent.getIntExtra(NetworkUtils.STATUS_CODE, 0),
                                intent.getStringExtra(NetworkUtils.ERROR_MESSAGE)
                        )
                );

            }

            LocalBroadcastManager
                    .getInstance(mContext)
                    .unregisterReceiver(mRegisterUserReceiver);

        }

    }

    private final class UserFitnessActivityReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {

            boolean successful = intent.getBooleanExtra(UserFitnessActivityService.SUCCESSFUL, false);

            if (successful) {

                mUserFitnessActivityCallbacks.onSuccess();

            } else {

                mUserFitnessActivityCallbacks.onFailure(
                        new BriteHealthException(
                                intent.getIntExtra(NetworkUtils.STATUS_CODE, 0),
                                intent.getStringExtra(NetworkUtils.ERROR_MESSAGE)
                        )
                );

            }

            LocalBroadcastManager
                    .getInstance(mContext)
                    .unregisterReceiver(mUserFitnessActivityReceiver);

        }

    }

    private final class NextAssessmentTimesReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {

            boolean successful = intent.getBooleanExtra(NextAssessmentTimesService.SUCCESSFUL, false);

            if (successful) {

                Bundle bundle = intent.getBundleExtra(EXTRA_NEXT_ASSESSMENT_TIMES);
                mNextAssessmentTimesCallbacks.onSuccess(bundle);

            } else {

                mNextAssessmentTimesCallbacks.onFailure(
                        new BriteHealthException(
                                intent.getIntExtra(NetworkUtils.STATUS_CODE, 0),
                                intent.getStringExtra(NetworkUtils.ERROR_MESSAGE)
                        )
                );

            }

            LocalBroadcastManager
                    .getInstance(mContext)
                    .unregisterReceiver(mNextAssessmentTimesReceiver);

        }

    }

    private final class ScoreReceiver extends BroadcastReceiver {

        @Override
        public void onReceive(Context context, Intent intent) {

            boolean successful = intent.getBooleanExtra(ScoreService.SUCCESSFUL, false);

            if (successful) {

                ArrayList<Score> scores = new ArrayList<>();
                for (int i = 0; i < intent.getParcelableArrayListExtra(EXTRA_SCORES).size(); i++) {
                    scores.add((Score) intent.getParcelableArrayListExtra(EXTRA_SCORES).get(i));
                }

                mScoreCallbacks.onSuccess(scores);

            } else {

                mScoreCallbacks.onFailure(
                        new BriteHealthException(
                                intent.getIntExtra(NetworkUtils.STATUS_CODE, 0),
                                intent.getStringExtra(NetworkUtils.ERROR_MESSAGE)
                        )
                );

            }

            LocalBroadcastManager
                    .getInstance(mContext)
                    .unregisterReceiver(mScoreReceiver);

        }

    }

    /**
     * Builder class for {@link BriteHealthSdk} objects.
     * Provides a convenient way to set the various fields of a {@link BriteHealthSdk}.
     */
    public static final class Builder {

        private Context mContext;
        private String mApiKey;
        private String mSecret;

        public Builder(Context context) {

            mContext = context.getApplicationContext();

            try {

                ApplicationInfo applicationInfo = context.getPackageManager()
                        .getApplicationInfo(
                                context.getPackageName(),
                                PackageManager.GET_META_DATA
                        );

                Bundle metaData = applicationInfo.metaData;
                mApiKey = metaData.getString(EXTRA_API_KEY);
                mSecret = metaData.getString(EXTRA_DATA_SECRET);

            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            }

        }

        public Builder setCredentials(String apiKey, String secret) {
            mApiKey = apiKey;
            mSecret = secret;
            return this;
        }

        public BriteHealthSdk build() {
            return new BriteHealthSdk(this);
        }

    }

    /**
     * Provides callbacks that are called when the Brite Health server responds to
     * {@link #registerUser(BriteHealthUser, RegisterUserCallbacks)}.
     */
    public interface RegisterUserCallbacks {

        void onSuccess(BriteHealthUser briteHealthUser);
        void onFailure(BriteHealthException e);

    }

    /**
     * Provides callbacks that are called when the Brite Health server responds to
     * {@link #addUserFitnessActivity(FitnessActivity, UserFitnessActivityCallbacks)}.
     */
    public interface UserFitnessActivityCallbacks {

        void onSuccess();
        void onFailure(BriteHealthException e);

    }

    /**
     * Provides callbacks that are called when the Brite Health server responds to
     * {@link #getNextAssessmentTimes(NextAssessmentTimesCallbacks)}.
     */
    public interface NextAssessmentTimesCallbacks {

        void onSuccess(Bundle bundle);
        void onFailure(BriteHealthException e);

    }

    /**
     * Provides callbacks that are called when the Brite Health server responds to
     * {@link #getScore(int, int, long, long, ScoreCallbacks)}.
     */
    public interface ScoreCallbacks {

        void onSuccess(ArrayList<Score> scores);

        void onFailure(BriteHealthException e);

    }

    /**
     * User's gender as defined by Brite Health.
     */
    public enum Gender {
        N, M, F
    }

    /**
     * Different types of Analytics reports provided by Brite Health.
     */
    public enum AnalyticsType {
        INDIVIDUAL, COMPARATIVE
    }

}
