package ch.epfl.bbp.uima.utils;

import static ch.epfl.bbp.DateUtils.nowToHuman;
import static com.google.common.collect.Sets.newHashSet;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.uima.conceptMapper.ConceptMapper;

import ch.epfl.bbp.DateUtils;

/**
 * Create XML concept files (for {@link ConceptMapper})
 * 
 * @author renaud.richardet@epfl.ch
 */
public class ConceptFileWriter {

    private ConceptFileWriter() {// nope
    }

    /** A bean representing a lexicon context */
    public static class Concept {
        String canonical;
        String id;
        Set<String> variants;

        public Concept(String canonical, String id, Set<String> variants) {
            this.canonical = canonical;
            this.id = id;
            this.variants = variants;
        }

        public Set<String> getVariants() {
            return variants;
        }

        public Concept addVariant(String variant) {
            if (variants == null)
                variants = newHashSet();
            variants.add(variant);
            return this;
        }

        public String getCanonical() {
            return canonical;
        }

        public String getId() {
            return id;
        }

        @Override
        public String toString() {
            return id + ":" + canonical + " "
                    + Arrays.toString(variants.toArray()) + "";
        }
    }

    public static void writeConceptFile(File f, Collection<Concept> concepts)
            throws FileNotFoundException {
        writeConceptFile(f, concepts, "");
    }

    public static void writeConceptFile(File f, Collection<Concept> concepts,
            String msg) throws FileNotFoundException {

        PrintWriter w = new PrintWriter(f);
        w.append("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
        w.append("<!-- generated by ConceptFileWriter on " + nowToHuman()
                + " -->\n");
        if (msg != null && msg.length() > 0) {
            w.append("<!--  " + escape(msg) + " -->\n");
        }
        w.append("<synonym>\n");

        for (Concept c : concepts) {

            w.format("<token canonical=\"%s\" ref_id=\"%s\">\n",
                    escape(c.canonical), escape(c.id));

            for (String v : c.variants) {
                w.format("  <variant base=\"%s\" />\n", escape(v));
            }
            w.append("</token>\n");
        }
        w.append("</synonym>\n");
        w.close();
    }

    private static String escape(Object in) {
        return StringEscapeUtils.escapeXml(in.toString());
    }
}
