/*
 *  Copyright 2016
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.acra.collector;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.google.auto.service.AutoService;

import org.acra.ReportField;
import org.acra.builder.ReportBuilder;
import org.acra.config.CoreConfiguration;
import org.acra.data.CrashReportData;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

/**
 * Collects the holy stacktrace
 *
 * @author F43nd1r
 * @since 4.9.1
 */
@AutoService(Collector.class)
public final class StacktraceCollector extends BaseReportFieldCollector {
    public StacktraceCollector() {
        super(ReportField.STACK_TRACE, ReportField.STACK_TRACE_HASH);
    }

    @NonNull
    @Override
    public Order getOrder() {
        return Order.FIRST;
    }

    @Override
    void collect(@NonNull ReportField reportField, @NonNull Context context, @NonNull CoreConfiguration config, @NonNull ReportBuilder reportBuilder, @NonNull CrashReportData target) {
        switch (reportField) {
            case STACK_TRACE:
                target.put(ReportField.STACK_TRACE, getStackTrace(reportBuilder.getMessage(), reportBuilder.getException()));
                break;
            case STACK_TRACE_HASH:
                target.put(ReportField.STACK_TRACE_HASH, getStackTraceHash(reportBuilder.getException()));
                break;
            default:
                //will not happen if used correctly
                throw new IllegalArgumentException();
        }
    }

    @Override
    boolean shouldCollect(@NonNull Context context, @NonNull CoreConfiguration config, @NonNull ReportField collect, @NonNull ReportBuilder reportBuilder) {
        return collect == ReportField.STACK_TRACE || super.shouldCollect(context, config, collect, reportBuilder);
    }

    @NonNull
    private String getStackTrace(@Nullable String msg, @Nullable Throwable th) {
        final Writer result = new StringWriter();
        try (final PrintWriter printWriter = new PrintWriter(result)) {
            if (msg != null && !TextUtils.isEmpty(msg)) {
                printWriter.println(msg);
            }
            if (th != null) {
                th.printStackTrace(printWriter);
            }
            return result.toString();
        }
    }

    @NonNull
    private String getStackTraceHash(@Nullable Throwable th) {
        final StringBuilder res = new StringBuilder();
        Throwable cause = th;
        while (cause != null) {
            final StackTraceElement[] stackTraceElements = cause.getStackTrace();
            for (final StackTraceElement e : stackTraceElements) {
                res.append(e.getClassName());
                res.append(e.getMethodName());
            }
            cause = cause.getCause();
        }

        return Integer.toHexString(res.toString().hashCode());
    }
}
