/*
 * Decompiled with CFR 0.152.
 */
package org.hl7.fhir.validation.instance.utils;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.util.Base64;
import com.nimbusds.jose.util.X509CertUtils;
import java.io.ByteArrayInputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateScanner {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(CertificateScanner.class);
    private final CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
    private final ObjectMapper objectMapper = new ObjectMapper();

    public CertificateResult findCertificateByKid(Map<String, byte[]> loadedCertificates, List<String> foldersToScan, String targetKid) {
        CertificateResult result = this.scanLoadedCertificates(loadedCertificates, targetKid);
        if (result != null) {
            return result;
        }
        if (foldersToScan != null) {
            for (String folderPath : foldersToScan) {
                result = this.scanFolderWithCaching(folderPath, loadedCertificates, targetKid);
                if (result == null) continue;
                return result;
            }
        }
        return null;
    }

    private CertificateResult scanLoadedCertificates(Map<String, byte[]> loadedCertificates, String targetKid) {
        if (loadedCertificates == null || loadedCertificates.isEmpty()) {
            return null;
        }
        for (Map.Entry<String, byte[]> entry : loadedCertificates.entrySet()) {
            String filename = entry.getKey();
            byte[] fileBytes = entry.getValue();
            log.info("Scanning loaded certificate: " + filename);
            CertificateResult result = this.scanFileBytes(filename, fileBytes, targetKid);
            if (result == null) continue;
            return result;
        }
        return null;
    }

    private CertificateResult scanFolderWithCaching(String folderPath, Map<String, byte[]> loadedCertificates, String targetKid) {
        CertificateResult certificateResult;
        block9: {
            Path folder = Paths.get(folderPath, new String[0]);
            if (!Files.exists(folder, new LinkOption[0]) || !Files.isDirectory(folder, new LinkOption[0])) {
                log.warn("Invalid folder path: " + folderPath);
                return null;
            }
            Stream<Path> files = Files.walk(folder, new FileVisitOption[0]);
            try {
                certificateResult = files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(file -> this.scanFileWithCaching((Path)file, loadedCertificates, targetKid)).filter(Objects::nonNull).findFirst().orElse(null);
                if (files == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    log.error("Error scanning folder: " + e.getMessage());
                    return null;
                }
            }
            files.close();
        }
        return certificateResult;
    }

    private CertificateResult scanFileWithCaching(Path filePath, Map<String, byte[]> loadedCertificates, String targetKid) {
        String filename = filePath.toString();
        byte[] fileBytes = loadedCertificates.get(filename);
        if (fileBytes == null) {
            try {
                fileBytes = Files.readAllBytes(filePath);
                loadedCertificates.put(filename, fileBytes);
                log.info("Loaded and cached file: " + filename);
            }
            catch (Exception e) {
                log.error("Error reading file " + filename + ": " + e.getMessage());
                return null;
            }
        }
        return this.scanFileBytes(filename, fileBytes, targetKid);
    }

    private CertificateResult scanFileBytes(String filename, byte[] fileBytes, String targetKid) {
        String fileName = Paths.get(filename, new String[0]).getFileName().toString().toLowerCase();
        String extension = this.getFileExtension(fileName);
        log.info("Scanning file: " + filename);
        try {
            switch (extension) {
                case "json": 
                case "jwks": {
                    return this.scanJWKSBytes(filename, fileBytes, targetKid);
                }
                case "pem": 
                case "crt": 
                case "cer": {
                    return this.scanPEMBytes(filename, fileBytes, targetKid);
                }
                case "der": {
                    return this.scanDERBytes(filename, fileBytes, targetKid);
                }
                case "p12": 
                case "pfx": {
                    return this.scanPKCS12Bytes(filename, fileBytes, targetKid);
                }
                case "jks": {
                    return this.scanJKSBytes(filename, fileBytes, targetKid);
                }
            }
            return this.autoDetectAndScanBytes(filename, fileBytes, targetKid);
        }
        catch (Exception e) {
            log.error("Error processing file " + filename + ": " + e.getMessage());
            return null;
        }
    }

    private CertificateResult scanJWKSBytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        String content = new String(fileBytes);
        try {
            JWKSet jwkSet = JWKSet.parse((String)content);
            JWK jwk = jwkSet.getKeyByKeyId(targetKid);
            if (jwk != null) {
                log.info("Found matching JWK in JWKS file: " + filename);
                X509Certificate cert = this.extractCertificateFromJWK(jwk);
                return new CertificateResult(jwk, cert, filename);
            }
        }
        catch (Exception e) {
            try {
                JWK jwk = JWK.parse((String)content);
                if (targetKid.equals(jwk.getKeyID())) {
                    log.info("Found matching JWK in JSON file: " + filename);
                    X509Certificate cert = this.extractCertificateFromJWK(jwk);
                    return new CertificateResult(jwk, cert, filename);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return null;
    }

    private X509Certificate extractCertificateFromJWK(JWK jwk) {
        block7: {
            X509Certificate x509Certificate;
            List certChain = jwk.getX509CertChain();
            if (certChain == null || certChain.isEmpty()) break block7;
            byte[] certBytes = ((Base64)certChain.get(0)).decode();
            ByteArrayInputStream bis = new ByteArrayInputStream(certBytes);
            try {
                x509Certificate = (X509Certificate)this.certificateFactory.generateCertificate(bis);
            }
            catch (Throwable throwable) {
                try {
                    try {
                        bis.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            bis.close();
            return x509Certificate;
        }
        return null;
    }

    private CertificateResult scanPEMBytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        String pemContent = new String(fileBytes);
        List<X509Certificate> certificates = this.extractCertificatesFromPEM(pemContent);
        for (X509Certificate cert : certificates) {
            JWK jwk = this.createJWKFromCertificate(cert, targetKid);
            if (jwk == null) continue;
            log.info("Found matching certificate in PEM file: " + filename);
            return new CertificateResult(jwk, cert, filename);
        }
        return null;
    }

    private List<X509Certificate> extractCertificatesFromPEM(String pemContent) {
        Pattern certPattern = Pattern.compile("-----BEGIN CERTIFICATE-----\\s*([A-Za-z0-9+/\\s=]+)\\s*-----END CERTIFICATE-----", 40);
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        Matcher matcher = certPattern.matcher(pemContent);
        while (matcher.find()) {
            String base64Cert = matcher.group(1).replaceAll("\\s", "");
            try {
                X509Certificate cert = X509CertUtils.parse((String)base64Cert);
                if (cert == null) continue;
                certificates.add(cert);
            }
            catch (Exception e) {
                log.info("Failed to parse certificate block: " + e.getMessage());
            }
        }
        return certificates;
    }

    private CertificateResult scanDERBytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(fileBytes);){
            X509Certificate cert = (X509Certificate)this.certificateFactory.generateCertificate(bis);
            JWK jwk = this.createJWKFromCertificate(cert, targetKid);
            if (jwk != null) {
                log.info("Found matching certificate in DER file: " + filename);
                CertificateResult certificateResult = new CertificateResult(jwk, cert, filename);
                return certificateResult;
            }
        }
        return null;
    }

    private CertificateResult scanPKCS12Bytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        String[] passwords;
        for (String password : passwords = new String[]{"", "password", "changeit", "123456"}) {
            try {
                KeyStore keyStore = KeyStore.getInstance("PKCS12");
                try (ByteArrayInputStream bis = new ByteArrayInputStream(fileBytes);){
                    keyStore.load(bis, password.toCharArray());
                }
                CertificateResult result = this.scanKeyStore(keyStore, targetKid, filename);
                if (result == null) continue;
                log.info("Found matching certificate in PKCS12 file: " + filename);
                return result;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return null;
    }

    private CertificateResult scanJKSBytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        String[] passwords;
        for (String password : passwords = new String[]{"", "password", "changeit", "123456"}) {
            try {
                KeyStore keyStore = KeyStore.getInstance("JKS");
                try (ByteArrayInputStream bis = new ByteArrayInputStream(fileBytes);){
                    keyStore.load(bis, password.toCharArray());
                }
                CertificateResult result = this.scanKeyStore(keyStore, targetKid, filename);
                if (result == null) continue;
                log.info("Found matching certificate in JKS file: " + filename);
                return result;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return null;
    }

    private CertificateResult scanKeyStore(KeyStore keyStore, String targetKid, String source) throws Exception {
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            X509Certificate cert;
            JWK jwk;
            String alias = aliases.nextElement();
            if (!keyStore.isCertificateEntry(alias) || (jwk = this.createJWKFromCertificate(cert = (X509Certificate)keyStore.getCertificate(alias), targetKid)) == null) continue;
            return new CertificateResult(jwk, cert, source);
        }
        return null;
    }

    private CertificateResult autoDetectAndScanBytes(String filename, byte[] fileBytes, String targetKid) throws Exception {
        if (this.isProbablyText(fileBytes)) {
            String content = new String(fileBytes);
            if (content.trim().startsWith("{") || content.trim().startsWith("[")) {
                return this.scanJWKSBytes(filename, fileBytes, targetKid);
            }
            if (content.contains("-----BEGIN")) {
                return this.scanPEMBytes(filename, fileBytes, targetKid);
            }
        } else {
            try {
                return this.scanDERBytes(filename, fileBytes, targetKid);
            }
            catch (Exception e) {
                try {
                    return this.scanPKCS12Bytes(filename, fileBytes, targetKid);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        return null;
    }

    private JWK createJWKFromCertificate(X509Certificate cert, String targetKid) {
        try {
            String cn;
            String subjectDN;
            JWK jwk = JWK.parse((X509Certificate)cert);
            String certKid = jwk.getKeyID();
            if (targetKid.equals(certKid)) {
                return jwk;
            }
            String thumbprint = jwk.computeThumbprint().toString();
            if (targetKid.equals(thumbprint)) {
                return this.createJWKWithKid(jwk, targetKid);
            }
            String sha1Thumbprint = jwk.computeThumbprint("SHA-1").toString();
            if (targetKid.equals(sha1Thumbprint)) {
                return this.createJWKWithKid(jwk, targetKid);
            }
            String serialNumber = cert.getSerialNumber().toString(16);
            if (targetKid.equals(serialNumber)) {
                return this.createJWKWithKid(jwk, targetKid);
            }
            String serialDecimal = cert.getSerialNumber().toString(10);
            if (targetKid.equals(serialDecimal)) {
                return this.createJWKWithKid(jwk, targetKid);
            }
            byte[] skiExtension = cert.getExtensionValue("2.5.29.14");
            if (skiExtension != null && skiExtension.length > 4) {
                byte[] ski = Arrays.copyOfRange(skiExtension, 4, skiExtension.length);
                StringBuilder hexString = new StringBuilder();
                for (byte b : ski) {
                    hexString.append(String.format("%02x", b));
                }
                if (targetKid.equals(hexString.toString())) {
                    return this.createJWKWithKid(jwk, targetKid);
                }
                if (targetKid.equals(hexString.toString().toUpperCase())) {
                    return this.createJWKWithKid(jwk, targetKid);
                }
            }
            if ((subjectDN = cert.getSubjectX500Principal().getName()).contains("CN=") && targetKid.equals(cn = this.extractCN(subjectDN))) {
                return this.createJWKWithKid(jwk, targetKid);
            }
        }
        catch (Exception e) {
            log.error("Error creating JWK from certificate: " + e.getMessage());
        }
        return null;
    }

    private JWK createJWKWithKid(JWK originalJwk, String newKid) throws Exception {
        Map jwkMap = originalJwk.toJSONObject();
        jwkMap.put("kid", newKid);
        return JWK.parse((Map)jwkMap);
    }

    private String extractCN(String dn) {
        Pattern cnPattern = Pattern.compile("CN=([^,]+)");
        Matcher matcher = cnPattern.matcher(dn);
        if (matcher.find()) {
            return matcher.group(1).trim();
        }
        return null;
    }

    private boolean isProbablyText(byte[] bytes) {
        if (bytes.length == 0) {
            return false;
        }
        int textChars = 0;
        int sampleSize = Math.min(1024, bytes.length);
        for (int i = 0; i < sampleSize; ++i) {
            byte b = bytes[i];
            if ((b < 32 || b > 126) && b != 9 && b != 10 && b != 13) continue;
            ++textChars;
        }
        return (double)textChars / (double)sampleSize > 0.95;
    }

    private String getFileExtension(String fileName) {
        int lastDot = fileName.lastIndexOf(46);
        return lastDot > 0 ? fileName.substring(lastDot + 1) : "";
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            log.info("Usage: java CertificateScanner <folder_path1> [folder_path2...] <kid>");
            return;
        }
        String targetKid = args[args.length - 1];
        ArrayList<String> folderPaths = new ArrayList<String>();
        for (int i = 0; i < args.length - 1; ++i) {
            folderPaths.add(args[i]);
        }
        try {
            CertificateScanner scanner = new CertificateScanner();
            HashMap<String, byte[]> loadedCertificates = new HashMap<String, byte[]>();
            CertificateResult result = scanner.findCertificateByKid(loadedCertificates, folderPaths, targetKid);
            if (result != null) {
                log.info("\nFound matching certificate!");
                log.info("Source: " + result.getSource());
                if (result.hasJwk()) {
                    JWK jwk = result.getJwk();
                    log.info("Key ID: " + jwk.getKeyID());
                    log.info("Key Type: " + String.valueOf(jwk.getKeyType()));
                    log.info("Algorithm: " + String.valueOf(jwk.getAlgorithm()));
                    log.info("JWK: " + jwk.toJSONString());
                }
                if (result.hasCertificate()) {
                    X509Certificate cert = result.getCertificate();
                    log.info("\nX509 Certificate Details:");
                    log.info("Subject: " + cert.getSubjectX500Principal().getName());
                    log.info("Issuer: " + cert.getIssuerX500Principal().getName());
                    log.info("Serial Number: " + cert.getSerialNumber().toString(16));
                    log.info("Valid From: " + String.valueOf(cert.getNotBefore()));
                    log.info("Valid Until: " + String.valueOf(cert.getNotAfter()));
                    log.info("Signature Algorithm: " + cert.getSigAlgName());
                }
                log.info("\nLoaded certificates cache now contains " + loadedCertificates.size() + " files");
            } else {
                log.info("\nNo matching certificate found for kid: " + targetKid);
                log.info("Scanned " + loadedCertificates.size() + " files total");
            }
        }
        catch (Exception e) {
            log.error("Error: " + e.getMessage(), (Throwable)e);
        }
    }

    public static class CertificateResult {
        private final JWK jwk;
        private final X509Certificate certificate;
        private final String source;

        public CertificateResult(JWK jwk, X509Certificate certificate, String source) {
            this.jwk = jwk;
            this.certificate = certificate;
            this.source = source;
        }

        public JWK getJwk() {
            return this.jwk;
        }

        public X509Certificate getCertificate() {
            return this.certificate;
        }

        public String getSource() {
            return this.source;
        }

        public boolean hasJwk() {
            return this.jwk != null;
        }

        public boolean hasCertificate() {
            return this.certificate != null;
        }
    }
}

