/*
 * Copyright 2016 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.rio.dhcp;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class DhcpServerConfigurationEdit {

  private final DhcpServerConfiguration configuration;
  private final List<Command> commandList = new ArrayList<>();

  DhcpServerConfigurationEdit(DhcpServerConfiguration configuration) {
    this.configuration = configuration;
  }

  public DhcpServerConfigurationEdit addHost(Host host) {
    Objects.requireNonNull(host);
    commandList.add(new AddHost(host));
    return this;
  }

  public DhcpServerConfigurationEdit clear() {
    commandList.add(Clear.INSTANCE);
    return this;
  }

  public void commit() {
    synchronized (configuration) {
      configuration.lock();
      for (Command command : commandList) {
        command.execute(configuration);
      }
    }
  }

  private static class AddHost extends Command {

    private final Host host;

    public AddHost(Host host) {
      this.host = host;
    }

    @Override
    void execute(DhcpServerConfiguration configuration) {
      configuration.addHost(host);
    }

  }

  private static class Clear extends Command {

    static final Command INSTANCE = new Clear();

    private Clear() {
    }

    @Override
    void execute(DhcpServerConfiguration configuration) {
      configuration.clear();
    }

  }

  private static abstract class Command {

    abstract void execute(DhcpServerConfiguration configuration);

  }

}