/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi.eclipse;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;

import org.w3c.dom.*;
import org.xml.sax.SAXException;

/**
 * Parse the Eclipse project information for the classpath. Unfortunately, it is
 * impossible to read the variables. They are ignored but that can cause
 * problems.
 * 
 * @version $Revision: 1.1 $
 */
public class EclipseClasspath {
	static DocumentBuilderFactory	documentBuilderFactory	= DocumentBuilderFactory
																	.newInstance();
	DocumentBuilder					db;
	File							project;
	File							workspace;
	List							sources					= new ArrayList();
	List							classpath				= new ArrayList();
	List							dependents				= new ArrayList();
	File							output;

	/**
	 * Parse an Eclipse project structure to discover the classpath.
	 * 
	 * @param workspace Points to workspace
	 * @param project Points to project
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */

	public EclipseClasspath(File workspace, File project)
			throws ParserConfigurationException, SAXException, IOException {
		this.project = project;
		this.workspace = workspace;
		db = documentBuilderFactory.newDocumentBuilder();
		parse(project, true);
		db = null;
	}

	/**
	 * Recursive routine to parse the files. If a sub project is detected, it is
	 * parsed before the parsing continues. This should give the right order.
	 * 
	 * @param project Project directory
	 * @param top If this is the top project
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	void parse(File project, boolean top) throws ParserConfigurationException,
			SAXException, IOException {
		File file = new File(project, ".classpath");
		if (!file.exists())
			throw new IllegalArgumentException(".classpath file not found: "
					+ file.getAbsolutePath());

		File cp = new File(project, ".classpath");
		if (!cp.exists())
			return;

		Document doc = db.parse(cp);
		NodeList nodelist = doc.getDocumentElement().getElementsByTagName(
				"classpathentry");

		if (nodelist == null)
			throw new IllegalArgumentException(
					"Can not find classpathentry in classpath file");

		for (int i = 0; i < nodelist.getLength(); i++) {
			Node node = nodelist.item(i);
			NamedNodeMap attrs = node.getAttributes();
			String kind = get(attrs, "kind");
			if ("src".equals(kind)) {
				String path = get(attrs, "path");
				boolean exported = "true".equalsIgnoreCase(get(
						attrs,
						"exported"));
				if (path.startsWith("/")) {
					// We have another project
					path = path.replace('/', File.separatorChar);
					File subProject = new File(workspace, path.substring(1));
					dependents.add(subProject);
					if (top || exported)
						parse(subProject, false);
				}
				else {
					File src = new File(project, path);
					sources.add(src);
				}
			}
			else if ("lib".equals(kind)) {
				String path = get(attrs, "path");
				boolean exported = "true".equalsIgnoreCase(get(
						attrs,
						"exported"));
				if (top || exported) {
					File jar = null;
					path = path.replace('/', File.separatorChar);
					if (path.startsWith(File.separator))
						jar = new File(workspace, path.substring(1));
					else {
						File f = new File(path);
						if (f.isAbsolute()) {
							jar = f;
						}
						else {
							jar = new File(project, path);
						}
					}
					classpath.add(jar);
				}
			}
			else if ("output".equals(kind)) {
				String path = get(attrs, "path");
				path = path.replace('/', File.separatorChar);
				output = new File(project, path);
				classpath.add(output);
			}
		}
	}

	private String get(NamedNodeMap map, String name) {
		Node node = map.getNamedItem(name);
		if (node == null)
			return null;

		return node.getNodeValue();
	}

	public List getClasspath() {
		return classpath;
	}

	public List getSourcepath() {
		return sources;
	}

	public File getOutput() {
		return output;
	}

	public List getDependents() {
		return dependents;
	}
}
