/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.regex.*;
import java.util.zip.*;

public class Jar {
	Map			resources	= new TreeMap();
	Map			directories	= new TreeMap();
	Manifest	manifest;
	boolean		manifestFirst;
	String		name;
	File		source;

	public Jar(String name) {
		this.name = name;
	}

	public Jar(String name, File dirOrFile) throws ZipException, IOException {
		this(name);
		source = dirOrFile;
		if (dirOrFile.isDirectory())
			FileResource.build(this, dirOrFile, Analyzer.doNotCopy);
		else
			ZipResource.build(this, dirOrFile);
	}

	public Jar(String name, InputStream in) throws IOException {
		this(name);
		EmbeddedResource.build(this, in);
	}

	public Jar(String name, String path) throws IOException {
		this(name, new FileInputStream(path));
	}

	public Jar(File jar) throws IOException {
		this(jar.getName(), jar);
	}

	public void setName(String name) {
		this.name = name;
	}

	public String toString() {
		return "Jar:" + name;
	}

	public void putResource(String path, Resource resource) {
		if (path.equals("META-INF/MANIFEST.MF")) {
			manifest = null;
			if (resources.isEmpty())
				manifestFirst = true;
		}
		resources.put(path, resource);
		String dir = getDirectory(path);
		Map s = (Map) directories.get(dir);
		if (s == null) {
			s = new TreeMap();
			directories.put(dir, s);
		}
		s.put(path, resource);
	}

	public Resource getResource(String path) {
		return (Resource) resources.get(path);
	}

	private String getDirectory(String path) {
		int n = path.lastIndexOf('/');
		if (n < 0)
			return "";

		return path.substring(0, n);
	}

	public Map getDirectories() {
		return directories;
	}

	public Map getResources() {
		return resources;
	}

	public void addDirectory(Map directory) {
		for (Iterator c = directory.entrySet().iterator(); c.hasNext();) {
			Map.Entry entry = (Map.Entry) c.next();
			String key = (String) entry.getKey();
			if (!key.endsWith(".java"))
				putResource(key, (Resource) entry.getValue());

		}
	}

	public Manifest getManifest() throws IOException {
		if (manifest == null) {
			Resource manifestResource = getResource("META-INF/MANIFEST.MF");
			if (manifestResource != null)
				manifest = new Manifest(manifestResource.openInputStream());
		}
		return manifest;
	}

	public boolean exists(String path) {
		return resources.containsKey(path);
	}

	public void setManifest(Manifest manifest) {
		manifestFirst = true;
		this.manifest = manifest;
	}

	public void write(File file) throws Exception {
		try {
			OutputStream out = new FileOutputStream(file);
			write(out);
			out.close();
		} catch (Throwable t) {
			try {
				file.delete();
			} catch (Exception ee) {
				ee.printStackTrace();
			}
			t.printStackTrace();
		}
	}

	public void write(String file) throws Exception {
		write(new File(file));
	}

	public void write(OutputStream out) throws Exception {
		JarOutputStream jout = new JarOutputStream(out);
		Set done = new HashSet();

		Set directories = new HashSet();
		doManifest(done, jout);
		for (Iterator i = getResources().entrySet().iterator(); i.hasNext();) {
			Map.Entry entry = (Map.Entry) i.next();
			// Skip metainf contents
			if (!done.contains(entry.getKey()))
				writeResource(jout, directories, (String) entry.getKey(),
						(Resource) entry.getValue());
		}
		jout.close();
		out.close();
	}

	private void doManifest(Set done, JarOutputStream jout) throws Exception {
		JarEntry ze = new JarEntry("META-INF/MANIFEST.MF");
		jout.putNextEntry(ze);
		getManifest().write(jout);
		jout.closeEntry();
		done.add(ze.getName());
	}

	private void writeResource(JarOutputStream jout, Set directories,
			String path, Resource resource) throws IOException {
		if (resource == null)
			return;

		createDirectories(directories, jout, path);
		ZipEntry ze = new ZipEntry(path);
		ze.setMethod(ZipEntry.DEFLATED);
		jout.putNextEntry(ze);
		try {
			resource.write(jout);
		} catch (Exception e) {
			throw new IllegalArgumentException("Cannot write resource: " + path
					+ " " + e);
		}
		jout.closeEntry();
	}

	void createDirectories(Set directories, JarOutputStream zip, String name)
			throws IOException {
		int index = name.lastIndexOf('/');
		if (index > 0) {
			String path = name.substring(0, index);
			if (directories.contains(path))
				return;
			createDirectories(directories, zip, path);
			ZipEntry ze = new ZipEntry(path + '/');
			zip.putNextEntry(ze);
			zip.closeEntry();
			directories.add(path);
		}
	}

	public String getName() {
		return name;
	}

	/** 
	 * Add all the resources in the given jar that match the given filter.
	 * 
	 * @param sub the jar
	 * @param filter a pattern that should match the resoures in sub to be added
	 */
	public void addAll(Jar sub, Pattern filter) {
		for (Iterator r = sub.getResources().keySet().iterator(); r.hasNext();) {
			String name = (String) r.next();
			if ( filter == null || filter.matcher(name).matches())
				putResource(name, sub.getResource(name));			
		}
	}
}
