/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.util.*;

public class Clazz {
	static byte		SkipTable[]	= {0, // 0 non existent
			-1, // 1 CONSTANT_utf8 UTF 8, handled in
			// method
			-1, // 2
			4, // 3 CONSTANT_Integer
			4, // 4 CONSTANT_Float
			8, // 5 CONSTANT_Long (index +=2!)
			8, // 6 CONSTANT_Double (index +=2!)
			-1, // 7 CONSTANT_Class
			2, // 8 CONSTANT_String
			4, // 9 CONSTANT_FieldRef
			4, // 10 CONSTANT_MethodRef
			4, // 11 CONSTANT_InterfaceMethodRef
			4, // 12 CONSTANT_NameAndType
								};

	Map				imports		= new HashMap();
	String			path;
	boolean			activator;
	String			className;
	//static String	type		= "([BCDFIJSZ\\[]|L[^<>]+;)";
	//static Pattern	descriptor	= Pattern.compile("\\(" + type + "*\\)(("
	//									+ type + ")|V)");
	int				minor		= 0;
	int				major		= 0;

	public Clazz(String path) {
		this.path = path;
	}

	public Clazz(String path, InputStream in) throws IOException {
		this.path = path;
		DataInputStream din = new DataInputStream(in);
		parseClassFile(din);
	}

	Set parseClassFile(DataInputStream in) throws IOException {
		Set xref = new HashSet();
		Set classes = new HashSet();
		Set descriptors = new HashSet();
		Hashtable pool = new Hashtable();
		int magic = in.readInt();
		if (magic != 0xCAFEBABE)
			throw new IOException("Not a valid class file (no CAFEBABE header)");
		
		minor = in.readShort(); // minor version
		major = in.readShort(); // major version
		int count = in.readUnsignedShort();
		process: for (int i = 1; i < count; i++) {
			byte tag = in.readByte();
			switch (tag) {
				case 0 :
					break process;
				case 1 :
					// CONSTANT_Utf8
					String name = in.readUTF();
					xref.add(name);
					pool.put(new Integer(i), name);

					// Check if this could be a descriptor.
					// If so we assume it is.
					//if (name != null && name.length() > 0
					//		&& name.charAt(0) == '('
					//		&& descriptor.matcher(name).matches()) {
					//	parseDescriptor(name);
					//}
					break;
				// A Class constant is just a short reference in
				// the constant pool
				case 7 :
					// CONSTANT_Class
					Integer index = new Integer(in.readShort());
					classes.add(index);
					break;
				// For some insane optimization reason are
				// the long and the double two entries in the
				// constant pool. See 4.4.5
				case 5 :
					// CONSTANT_Long
				case 6 :
					// CONSTANT_Double
					in.skipBytes(8);
					i++;
					break;

				// Name and Type
				case 12 :
					in.skip(2); // Name index
					int descriptorIndex = in.readShort();
					descriptors.add(new Integer(descriptorIndex));
					break;

				// We get the skip count for each record type
				// from the SkipTable. This will also automatically
				// abort when
				default :
					if (tag == 2)
						throw new IOException("Invalid tag " + tag);
					in.skipBytes(SkipTable[tag]);
					break;
			}
		}

		/*
		 * Parse after the constant pool, code thanks to Hans Christian
		 * Falkenberg
		 */

		in.skip(2+2+2); // access flags, this_class CONSTANT_class, super class CONSTANT_class
		
		int interfacesCount = in.readUnsignedShort();
		in.skip( interfacesCount * 2 );

		int fieldsCount = in.readUnsignedShort();
		for (int i = 0; i < fieldsCount; i++) {
			in.skip(2+2); // skip access flags + name indexes
			int descriptorIndex = in.readUnsignedShort();
			descriptors.add(new Integer(descriptorIndex));
			skipAttributes(in);
		}

		int methodCount = in.readUnsignedShort();
		for (int i = 0; i < methodCount; i++) {
			in.skip(2+2); // skip access flags + name indexes
			int descriptorIndex = in.readUnsignedShort();
			descriptors.add(new Integer(descriptorIndex));
			skipAttributes(in);
		}

		// PARSING: Rest of class file is class file attributes.
		// Just ignoring them, they contain no (important)
		// package references.

		//
		// Now iterate over all classes we found and
		// parse those as well. We skip duplicates
		//

		for (Iterator e = classes.iterator(); e.hasNext();) {
			Integer n = (Integer) e.next();
			String next = (String) pool.get(n);
			checkActivator(next);
		}
		
		//
		// Parse all the descriptors we found
		//
		
		for (Iterator e = descriptors.iterator(); e.hasNext();) {
			Integer n = (Integer) e.next();
			String prototype = (String) pool.get(n);
			if (prototype != null)
				parseDescriptor(prototype);
		}
		return xref;
	}

	private void checkActivator(String next) {
		if (next != null) {
			String normalized = normalize(next);
			if (normalized != null) {
				// For purposes of trying to guess the activator class, we
				// assume
				// that any class that references the BundleActivator
				// interface
				// must be a BundleActivator implementation.
				if (normalized
						.startsWith("org/osgi/framework/BundleActivator")) {
					className = path.replace('/', '.');
					className = className.substring(0, className.length()
							- ".class".length());
					activator = true;
				}
				String pack = getPackage(normalized);
				packageReference(pack);
			}
		}
		else
			throw new IllegalArgumentException("Invalid class, parent=");
	}

	private void skipAttributes(DataInputStream in) throws IOException {
		int attributesCount = in.readUnsignedShort();
		// skip attributes
		for (int j = 0; j < attributesCount; j++) {
			// skip name CONSTANT_Utf8 pointer
			in.readUnsignedShort();
			// length
			int attributeLength = in.readInt();
			in.skip(attributeLength);
		}
	}

	void packageReference(String pack) {
		if (pack.indexOf('<') > 0)
			System.out.println("Oops: " + pack);
		if (!imports.containsKey(pack))
			imports.put(pack, new HashMap());
	}

	void parseDescriptor(String prototype) {
		addReference(prototype);
		StringTokenizer st = new StringTokenizer(prototype, "(;)", true);
		while (st.hasMoreTokens()) {
			if (st.nextToken().equals("(")) {
				String token = st.nextToken();
				while (!token.equals(")")) {
					addReference(token);
					token = st.nextToken();
				}
				token = st.nextToken();
				addReference(token);
			}
		}
	}

	private void addReference(String token) {
		while (token.startsWith("["))
			token = token.substring(1);

		if (token.startsWith("L")) {
			String clazz = normalize(token.substring(1));
			if (clazz.startsWith("java/"))
				return;
			String pack = getPackage(clazz);
			packageReference(pack);
		}
	}

	static String normalize(String s) {
		if (s.startsWith("[L"))
			return normalize(s.substring(2));
		if (s.startsWith("["))
			if (s.length() == 2)
				return null;
			else
				return normalize(s.substring(1));
		if (s.endsWith(";"))
			return normalize(s.substring(0, s.length() - 1));
		return s + ".class";
	}

	public static String getPackage(String clazz) {
		int n = clazz.lastIndexOf('/');
		if (n < 0)
			return ".";
		return clazz.substring(0, n).replace('/', '.');
	}

	public Map getReferred() {
		return imports;
	}

	String getClassName() {
		return className;
	}

	boolean isActivator() {
		return activator;
	}

	public String getPath() {
		return path;
	}

	public Set xref(InputStream in) throws IOException {
		DataInputStream din = new DataInputStream(in);
		return parseClassFile(din);
	}

}

/*
 * package aQute.lib.osgi;
 * 
 * import java.io.*; import java.util.*;
 * 
 * import org.objectweb.asm.*;
 * 
 * public class ClassNew implements ClassVisitor, MethodVisitor { Map imports =
 * new HashMap(); String path; boolean activator; String className;
 * 
 * public ClassNew(String path, InputStream in) throws IOException { this.path =
 * path; ClassReader rdr = new ClassReader(in); rdr.accept(this, true); }
 * 
 * String getClassName() { return className; }
 * 
 * 
 * public FieldVisitor visitField(int access, String name, String desc, String
 * signature, Object value) { addType(Type.getType(desc)); return null; }
 * 
 * public MethodVisitor visitMethod(int access, String name, String desc, String
 * signature, String[] exceptions) { addType(Type.getReturnType(desc)); Type
 * types[] = Type.getArgumentTypes(desc); for (int i = 0; i < types.length; i++) {
 * addType(types[i]); } addInternalNames(exceptions); return this; }
 * 
 * public void visitFieldInsn(int opcode, String owner, String name, String
 * desc) { addType(Type.getType(desc)); }
 * 
 * public void visitLocalVariable(String name, String desc, String signature,
 * Label start, Label end, int index) { addType(Type.getType(desc)); }
 * 
 * public void visitMethodInsn(int opcode, String owner, String name, String
 * desc) { addType(Type.getReturnType(desc)); Type types[] =
 * Type.getArgumentTypes(desc); for (int i = 0; i < types.length; i++) {
 * addType(types[i]); }
 * 
 * addInternalName(owner); }
 * 
 * public void visitMultiANewArrayInsn(String desc, int dims) {
 * addType(Type.getType(desc)); }
 * 
 * public void visitTypeInsn(int opcode, String desc) { if (
 * !desc.startsWith("[")) addInternalName(desc); else {
 * addType(Type.getType(desc)); } }
 * 
 * public void visit(int version, int access, String name, String signature,
 * String superName, String[] interfaces) { addInternalNames(interfaces);
 * addInternalName(superName); }
 * 
 * private void addInternalNames(String[] names) { if ( names == null ) return;
 * 
 * for (int i = 0; i < names.length; i++) { addInternalName(names[i]); } }
 * 
 * private void packageReference(String pack) { if ( pack.equals(".")) return;
 * 
 * if (!imports.containsKey(pack)) { imports.put(pack, new HashMap()); } }
 * 
 * private static String getPackage(String clazz) { int n =
 * clazz.lastIndexOf('.'); if (n < 0) return "."; return clazz.substring(0, n); }
 * 
 * 
 * private void addType(Type type) { String cname = type.getClassName();
 * packageReference(getPackage(cname)); }
 * 
 * private void addInternalName(String owner) { String c = owner.replace('/',
 * '.'); packageReference(getPackage(c)); } //
 * -------------------------------------------------- // Dummy methods, can be
 * ignored // --------------------------------------------------
 * 
 * public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
 * return null; }
 * 
 * public void visitAttribute(Attribute attr) { }
 * 
 * public void visitEnd() { }
 * 
 * public void visitInnerClass(String name, String outerName, String innerName,
 * int access) { }
 * 
 * public void visitOuterClass(String owner, String name, String desc) { }
 * 
 * public void visitSource(String source, String debug) { }
 * 
 * public AnnotationVisitor visitAnnotationDefault() { return null; }
 * 
 * public void visitCode() { }
 * 
 * public void visitIincInsn(int var, int increment) { }
 * 
 * public void visitInsn(int opcode) { }
 * 
 * public void visitIntInsn(int opcode, int operand) { }
 * 
 * public void visitJumpInsn(int opcode, Label label) { }
 * 
 * public void visitLabel(Label label) { }
 * 
 * public void visitLdcInsn(Object cst) { }
 * 
 * public void visitLineNumber(int line, Label start) { }
 * 
 * public void visitLookupSwitchInsn(Label dflt, int[] keys, Label[] labels) { }
 * 
 * public void visitMaxs(int maxStack, int maxLocals) { }
 * 
 * public AnnotationVisitor visitParameterAnnotation(int parameter, String desc,
 * boolean visible) { return null; }
 * 
 * public void visitTableSwitchInsn(int min, int max, Label dflt, Label[]
 * labels) { }
 * 
 * public void visitTryCatchBlock(Label start, Label end, Label handler, String
 * type) { }
 * 
 * public void visitVarInsn(int opcode, int var) { }
 * 
 * public Map getReferred() { return imports; }
 * 
 * public String getPath() { return path; } }
 * 
 */

