/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.regex.*;
import java.util.zip.*;

/**
 * Include-Resource: ( [name '=' ] file )+
 * 
 * Private-Package: package-decl ( ',' package-decl )*
 * 
 * Export-Package: package-decl ( ',' package-decl )*
 * 
 * Import-Package: package-decl ( ',' package-decl )*
 * 
 * @version $Revision: 1.4 $
 */
public class Builder extends Analyzer {
	boolean			sources	= false;
	private File[]	sourcePath;

	public Jar build() throws Exception {
		sources = getProperty("-sources") != null;

		dot = new Jar("dot");
		doExpand(dot);
		doIncludeResources(dot);
		dot.setManifest(calcManifest());
		// This must happen after we analyzed so
		// we know what it is on the classpath
		addSources(dot);
		if (getProperty("-pom") != null)
			doPom(dot);

		doVerify(dot);
		if (dot.getResources().isEmpty())
			error("The JAR is empty");

		// Signer signer = getSigner();
		// if ( signer != null ) {
		// signer.signJar(dot);
		// }
		return dot;
	}

	/**
	 * 
	 */
	private void addSources(Jar dot) {
		if (!sources)
			return;

		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			getProperties().store(out, "Generated by BND, at " + new Date());
			dot.putResource("OSGI-OPT/bnd.bnd", new EmbeddedResource(out
					.toByteArray()));
		} catch (Exception e) {
			error("Can not embed bnd file in JAR: " + e);
		}

		for (Iterator cpe = classspace.keySet().iterator(); cpe.hasNext();) {
			String path = (String) cpe.next();
			path = path.substring(0, path.length() - ".class".length())
					+ ".java";

			for (int i = 0; i < sourcePath.length; i++) {
				File root = sourcePath[i];
				File f = new File(root, path);
				if (f.exists()) {
					dot
							.putResource("OSGI-OPT/src/" + path,
									new FileResource(f));
				}
			}
		}
	}

	private void doVerify(Jar dot) throws Exception {
		Verifier verifier = new Verifier(dot);
		verifier.verify();
		errors.addAll(verifier.getErrors());
		warnings.addAll(verifier.getWarnings());
	}

	private void doExpand(Jar jar) throws IOException {
		Map prive = replaceWithPattern(getHeader(Analyzer.PRIVATE_PACKAGE));
		Map export = replaceWithPattern(getHeader(Analyzer.EXPORT_PACKAGE));
		if (prive.isEmpty() && export.isEmpty()) {
			warnings
					.add("Neither Export-Package nor Private-Package is set, therefore no packages will be included");
		}
		doExpand(jar, "Export-Package ", export);
		doExpand(jar, "Private-Package ", prive);
	}

	private void doExpand(Jar jar, String name, Map instructions) {
		Set superfluous = new HashSet(instructions.keySet());
		for (Iterator c = classpath.iterator(); c.hasNext();) {
			Jar now = (Jar) c.next();
			doExpand(jar, instructions, now, superfluous);
		}
		if (superfluous.size() > 0) {
			StringBuffer sb = new StringBuffer();
			String del = "Instructions for " + name + " that are never used: ";
			for (Iterator i = superfluous.iterator(); i.hasNext();) {
				Instruction p = (Instruction) i.next();
				sb.append(del);
				sb.append(p.getPattern());
				del = ", ";
			}
			warning(sb.toString());
		}
	}

	/**
	 * Iterate over each directory in the class path entry and check if that
	 * directory is a desired package.
	 * 
	 * @param included
	 * @param classpathEntry
	 */
	private void doExpand(Jar jar, Map included, Jar classpathEntry,
			Set superfluous) {
		for (Iterator p = classpathEntry.getDirectories().entrySet().iterator(); p
				.hasNext();) {
			Map.Entry directory = (Map.Entry) p.next();
			String path = (String) directory.getKey();

			if (doNotCopy.matcher(getName(path)).matches())
				continue;

			String pack = path.replace('/', '.');
			Instruction instr = matches(included, pack, superfluous);
			if (instr != null && !instr.isNegated()) {
				Map contents = (Map) directory.getValue();
				jar.addDirectory(contents);
			}
		}
	}

	private Map replaceWithPattern(Map header) {
		Map map = new LinkedHashMap();
		for (Iterator e = header.entrySet().iterator(); e.hasNext();) {
			Map.Entry entry = (Map.Entry) e.next();
			String pattern = (String) entry.getKey();
			Instruction instr = Instruction.getPattern(pattern);
			map.put(instr, entry.getValue());
		}
		return map;
	}

	private Instruction matches(Map instructions, String pack,
			Set superfluousPatterns) {
		for (Iterator i = instructions.keySet().iterator(); i.hasNext();) {
			Instruction pattern = (Instruction) i.next();
			if (pattern.matches(pack)) {
				superfluousPatterns.remove(pattern);
				return pattern;
			}
		}
		return null;
	}

	private Map getHeader(String string) {
		if (string == null)
			return new LinkedHashMap();
		return parseHeader(getProperty(string));
	}

	/**
	 * Parse the Bundle-Includes header. Files in the bundles Include header are
	 * included in the jar. The source can be a directory or a file.
	 * 
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	private void doIncludeResources(Jar jar) throws Exception {
		Macro macro = new Macro(getProperties(), this);

		String includes = getProperty("Bundle-Includes");
		if (includes == null)
			includes = getProperty("Include-Resource");
		else
			warnings
					.add("Please use Include-Resource instead of Bundle-Includes");

		if (includes == null)
			return;

		for (Iterator i = getClauses(includes).iterator(); i.hasNext();) {
			boolean preprocess = false;
			String clause = (String) i.next();
			if (clause.startsWith("{") && clause.endsWith("}")) {
				preprocess = true;
				clause = clause.substring(1, clause.length() - 1).trim();
			}

			if (clause.startsWith("@")) {
				extractFromJar(jar, clause.substring(1));
			} else {
				String parts[] = clause.split("\\s*=\\s*");

				File source;
				String destinationPath;

				if (parts.length == 1) {
					// Just a copy, destination path defined by
					// source path.
					source = new File(base, parts[0]);

					// Directories should be copied to the root
					// but files to their file name ...
					if (source.isDirectory())
						destinationPath = "";
					else
						destinationPath = source.getName();
				} else {
					source = new File(base, parts[1]);
					destinationPath = parts[0];
				}

				// Some people insist on ending a directory with
				// a slash ... it now also works if you do /=dir
				if (destinationPath.endsWith("/"))
					destinationPath = destinationPath.substring(0,
							destinationPath.length() - 1);

				copy(jar, destinationPath, source, preprocess ? macro : null);
			}
		}
	}

	/**
	 * Extra resources from a Jar and add them to the given jar. The clause is
	 * the
	 * 
	 * @param jar
	 * @param clauses
	 * @param i
	 * @throws ZipException
	 * @throws IOException
	 */
	private void extractFromJar(Jar jar, String name) throws ZipException,
			IOException {
		// Inline all resources and classes from another jar
		// optionally appended with a modified regular expression
		// like @zip.jar!/META-INF/MANIFEST.MF
		int n = name.lastIndexOf("!/");
		Pattern filter = null;
		if (n > 0) {
			String fstring = name.substring(n + 2);
			name = name.substring(0, n);
			filter = wildcard(fstring);
		}
		Jar sub = getJarFromName(name, "extract from jar");
		jar.addAll(sub, filter);
	}

	private Pattern wildcard(String spec) {
		StringBuffer sb = new StringBuffer();
		for (int j = 0; j < spec.length(); j++) {
			char c = spec.charAt(j);
			switch (c) {
			case '.':
				sb.append("\\.");
				break;

			case '*':
				// test for ** (all directories)
				if (j < spec.length() - 1 && spec.charAt(j + 1) == '*') {
					sb.append(".*");
					j++;
				} else
					sb.append("[^/]*");
				break;
			default:
				sb.append(c);
				break;
			}
		}
		String s = sb.toString();
		try {
			return Pattern.compile(s);
		} catch (Exception e) {
			error("Invalid regular expression on wildcarding: " + spec
					+ " used *");
		}
		return null;
	}

	private void copy(Jar jar, String path, File from, Macro macro)
			throws Exception {
		if (doNotCopy.matcher(from.getName()).matches())
			return;

		if (from.isDirectory()) {
			String next = path;
			if (next.length() != 0)
				next += '/';

			File files[] = from.listFiles();
			for (int i = 0; i < files.length; i++) {
				copy(jar, next + files[i].getName(), files[i], macro);
			}
		} else {
			if (macro != null) {
				String content = read(from);
				content = macro.process(content);
				jar.putResource(path, new EmbeddedResource(content
						.getBytes("UTF-8")));
			} else
				jar.putResource(path, new FileResource(from));
		}
	}

	private String read(File from) throws Exception {
		long size = from.length();
		byte[] buffer = new byte[(int) size];
		FileInputStream in = new FileInputStream(from);
		in.read(buffer);
		in.close();
		return new String(buffer, "UTF-8");
	}

	private String getName(String where) {
		int n = where.lastIndexOf('/');
		if (n < 0)
			return where;

		return where.substring(n + 1);
	}

	public void setSourcepath(File[] files) {
		sourcePath = files;
	}

	/**
	 * Create a POM reseource for Maven containing as much information as
	 * possible from the manifest.
	 * 
	 * @param output
	 * @param builder
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public void doPom(Jar dot) throws FileNotFoundException, IOException {
		{
			Manifest manifest = dot.getManifest();
			String name = manifest.getMainAttributes().getValue(
					Analyzer.BUNDLE_NAME);
			String description = manifest.getMainAttributes().getValue(
					Analyzer.BUNDLE_DESCRIPTION);
			String docUrl = manifest.getMainAttributes().getValue(
					Analyzer.BUNDLE_DOCURL);
			String version = manifest.getMainAttributes().getValue(
					Analyzer.BUNDLE_VERSION);
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			PrintStream ps = new PrintStream(s);
			String bsn = manifest.getMainAttributes().getValue(
					Analyzer.BUNDLE_SYMBOLICNAME);
			String licenses = manifest.getMainAttributes().getValue(BUNDLE_LICENSE);
			
			if (bsn == null) {
				errors
						.add("Can not create POM unless Bundle-SymbolicName is set");
				return;
			}

			bsn = bsn.trim();
			int n = bsn.lastIndexOf('.');
			if (n <= 0) {
				errors
						.add("Can not create POM unless Bundle-SymbolicName contains a .");
				return;
			}
			String groupId = bsn.substring(0,n);
			String artifactId = bsn.substring(n+1);
			ps
					.println("<project xmlns='http://maven.apache.org/POM/4.0.0' xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance' xsi:schemaLocation='http://maven.apache.org/POM/4.0.0 http://maven.apache.org/maven-v4_0_0.xsd'>");
			ps.println("  <modelVersion>4.0.0</modelVersion>");
			ps.println("  <groupId>" + groupId + "</groupId>");
			ps.println("  <artifactId>" + artifactId + "</artifactId>");
			ps.println("  <version>" + version + "</version>");
			if (description != null) {
				ps.println("  <description>");
				ps.print("    ");
				ps.println(description);
				ps.println("  </description>");
			}
			if (name != null) {
				ps.print("  <name>");
				ps.print(name);
				ps.println("</name>");
			}
			if (docUrl != null) {
				ps.print("  <url>");
				ps.print(docUrl);
				ps.println("</url>");
			}
			if (licenses != null) {
				ps.println("  <licenses>");
				String l[] = licenses.split("\\s*,\\s*");
				for (int i = 0; i < l.length; i++) {
					ps.println("    <license>");
					ps.print("      <url>");
					ps.print(  l[i]);
					ps.println("</url>");
					ps.println("    </license>");
				}
				ps.println("  </licenses>");
			}
			ps.println("</project>");
			ps.close();
			s.close();
			dot.putResource("pom.xml", new EmbeddedResource(s.toByteArray()));
		}
	}

}
