package asia.redact.bracket.properties;

import java.io.Serializable;
import java.nio.charset.Charset;

/**
 * This is like a content or mime type from an http page
 * 
 * There does not seem to be an official java properties content type. I am using text/x-java-properties
 * as it is cited here: http://stackoverflow.com/questions/6248130/what-is-the-mime-type-for-properties-files
 * 
 * Our default encoding is UTF-8 but for Compatibility you would use US-ASCII or ISO-8895-1
 * 
 * @author Dave
 *
 */
public class ContentType implements Serializable {

	private static final long serialVersionUID = 1L;
	private String mimeType; // normally text/x-java-properties
	private String charsetName; //java.nio charset name
	
	public ContentType() {
		mimeType = "text/x-java-properties";
		charsetName = "UTF-8";
	}
	
	public ContentType(String mimeType, String charsetName) {
		this.mimeType = mimeType;
		this.charsetName = charsetName;
	}
	
	/**
	 * <pre>
	 * The defined content types which will successfully validate are:
	 * text/x-java-properties
	 * text/x-java-dot-properties
	 * 
	 * </pre>
	 * @return
	 */
	public boolean validate() {
		if(mimeType == null || mimeType.equals("")) return false;
		if(charsetName == null || charsetName.equals("")) return false;
		try {
			Charset.forName(charsetName);
		}catch(Exception x){
			return false;
		}
		
		if(!((mimeType.equals("text/x-java-properties")||mimeType.equals("text/x-java-dot-properties")))){
			return false;
		}
		
		return true;
	}

	public String getMimeType() {
		return mimeType;
	}

	public void setMimeType(String mimeType) {
		this.mimeType = mimeType;
	}

	public String getCharsetName() {
		return charsetName;
	}

	public void setCharsetName(String charsetName) {
		this.charsetName = charsetName;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((charsetName == null) ? 0 : charsetName.hashCode());
		result = prime * result
				+ ((mimeType == null) ? 0 : mimeType.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ContentType other = (ContentType) obj;
		if (charsetName == null) {
			if (other.charsetName != null)
				return false;
		} else if (!charsetName.equals(other.charsetName))
			return false;
		if (mimeType == null) {
			if (other.mimeType != null)
				return false;
		} else if (!mimeType.equals(other.mimeType))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "content=\""+mimeType+"; charset="+charsetName+"\"";
	}
	
	
	public static ContentType getCompatibilityContentType() {
		return new ContentType("text/x-java-properties","ISO-8895-1");
	}
	
	public static ContentType getDotPropertiesContentType() {
		return new ContentType("text/x-java-dot-properties","UTF-8");
	}

}
