package app.raybritton.tokenstorage

import app.raybritton.tokenstorage.crypto.Crypto
import app.raybritton.tokenstorage.keyCrypto.KeyCrypto
import app.raybritton.tokenstorage.persistence.Persistence
import io.reactivex.BackpressureStrategy
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.subjects.ReplaySubject

class RxTokenStorage<C : Crypto, P : Persistence, KC : KeyCrypto>(
        crypto: C,
        persistence: P,
        keyCrypto: KC) {
    private val storage = TokenStorage(crypto, persistence, keyCrypto)

    private val loadSubjects = mutableMapOf<String, ReplaySubject<Optional<String>>>()
    private val containSubjects = mutableMapOf<String, ReplaySubject<Boolean>>()

    private val keysSubject = ReplaySubject.create<List<String>>(1)

    val crypto = storage.crypto
    val persistence = storage.persistence
    val keyCrypto = storage.keyCrypto

    fun clearAll(): Completable {
        return Completable.fromCallable {
            storage.clearAll()
            loadSubjects.values.forEach {
                it.onNext(Optional.None)
            }
            containSubjects.values.forEach {
                it.onNext(false)
            }
            keysSubject.onNext(listOf())
        }
    }

    fun clear(key: String): Completable {
        return Completable.fromCallable {
            storage.clear(key)
            reload(key)
        }
    }

    fun save(key: String, plaintext: String): Completable {
        return Completable.fromCallable {
            storage.save(key, plaintext)
            reload(key)
        }
    }

    fun load(key: String): Flowable<Optional<String>> {
        storage.crypto.verify()
        storage.keyCrypto.verify()
        if (loadSubjects.containsKey(key)) {
            return loadSubjects[key]!!.share().toFlowable(BackpressureStrategy.LATEST)
        } else {
            val subject = ReplaySubject.createWithSize<Optional<String>>(1)
            loadSubjects[key] = subject
            subject.onNext(Optional.auto(loadString(key)))
            return subject.share().toFlowable(BackpressureStrategy.LATEST)
        }
    }

    fun keys(): Flowable<List<String>> {
        storage.crypto.verify()
        storage.keyCrypto.verify()
        return keysSubject.share().toFlowable(BackpressureStrategy.LATEST)
                .startWith(storage.keys())
    }

    fun contains(key: String): Flowable<Boolean> {
        storage.crypto.verify()
        storage.keyCrypto.verify()
        if (containSubjects.containsKey(key)) {
            return containSubjects[key]!!.share().toFlowable(BackpressureStrategy.LATEST)
        } else {
            val subject = ReplaySubject.createWithSize<Boolean>(1)
            containSubjects[key] = subject
            subject.onNext(loadContains(key))
            return subject.share().toFlowable(BackpressureStrategy.LATEST)
        }
    }

    /**
     * Removes all files, etc generated by any crypto or persistence where possible
     * It also invalidates this instance of TokenStorage, continuing to use it will cause
     * undefined behaviour
     */
    fun reset(): Completable {
        return Completable.fromCallable {
           resetSync()
       }
    }

    fun clearAllSync() {
        storage.clearAll()
        loadSubjects.values.forEach {
            it.onNext(Optional.None)
        }
        containSubjects.values.forEach {
            it.onNext(false)
        }
        keysSubject.onNext(listOf())
    }

    fun clearSync(key: String) {
        storage.clear(key)
        reload(key)
    }

    fun saveSync(key: String, plaintext: String) {
        storage.save(key, plaintext)
        reload(key)
    }

    fun loadSync(key: String): String? {
        return storage.load(key)
    }

    fun containsSync(key: String): Boolean {
        return storage.contains(key)
    }

    /**
     * Removes all files, etc generated by any crypto or persistence where possible
     * It also invalidates this instance of TokenStorage, continuing to use it will cause
     * undefined behaviour
     */
    fun resetSync() {
        crypto.reset()
        persistence.reset()
    }

    fun wrap(key: String): RxTokenWrapper {
        return RxTokenWrapper(this, key)
    }

    private fun loadContains(key: String): Boolean {
        return storage.contains(key)
    }

    private fun loadString(key: String): String? {
        return storage.load(key)
    }

    private fun reload(key: String) {
        loadSubjects[key]?.onNext(Optional.auto(loadString(key)))
        containSubjects[key]?.onNext(loadContains(key))
        if (!keyCrypto.scramblesKeys()) {
            keysSubject.onNext(storage.keys())
        }
    }
}