package app.raybritton.tokenstorage.crypto

import java.io.File
import java.util.UUID

/**
 * Uses a passphrase (string) to encrypt and decrypt the strings, for example this could be
 * a PIN entered by the user. (AES/CBC/PKCS5Padding)
 *
 * Important:
 * passphrase MUST be set before calling any method
 *
 * A salt is generated when using this class, it is saved a file called 'crypto-salt' (by default) in
 * the apps files dir, if this is deleted or changed the stored data will be lost
 */
class PassphraseCrypto(private val saltDir: File,
                       private val saltFilename: String = "crypto-salt") : Crypto {
    private var secretKeys: AesCbcWithIntegrity.SecretKeys? = null

    /**
     * This can be set at any point, any number of times
     * When changed the crypto is rebuilt, if the passphrase does not match the
     * passphrase used to encrypt the data it will be mangled when decrypting but not lost
     */
    var passphrase: String? = null
        set(value) {
            if (field == value) return
            field = value
            invalidate()
        }

    private val salt by lazy {
        val saltFile = File(saltDir, saltFilename)
        if (saltFile.exists()) {
            saltFile.readLines()[0]
        } else {
            val newSalt = UUID.randomUUID().toString()
            saltFile.writeText(newSalt)
            newSalt
        }
    }

    private fun invalidate() {
        if (passphrase == null) {
            secretKeys = null
        } else {
            secretKeys = AesCbcWithIntegrity.generateKeyFromPassword(passphrase!!, salt)
        }
    }

    override fun encrypt(plaintext: String): String {
        return AesCbcWithIntegrity.encrypt(plaintext, secretKeys!!).toString()
    }

    override fun decrypt(encrypted: String): String {
        val mac = AesCbcWithIntegrity.CipherTextIvMac(encrypted)
        return String(AesCbcWithIntegrity.decrypt(mac, secretKeys!!))
    }

    override fun verify() {
        if (passphrase == null || secretKeys == null) {
            throw IllegalStateException("passphrase must be set before use")
        }
        if (passphrase.isNullOrEmpty()) {
            throw IllegalStateException("passphrase must have content")
        }
    }
}