/*
 * Decompiled with CFR 0.152.
 */
package androidx.security.app.authenticator;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.os.Build;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.collection.ArrayMap;
import androidx.collection.LruCache;
import androidx.security.app.authenticator.AppAuthenticatorUtils;
import androidx.security.app.authenticator.AutoValue_AppSignatureVerifier_AppSigningInfo;
import androidx.security.app.authenticator.AutoValue_AppSignatureVerifier_CacheEntry;
import com.google.auto.value.AutoValue;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

class AppSignatureVerifier {
    private static final String TAG = AppSignatureVerifier.class.getSimpleName();
    private static final String EXPECTED_IDENTITY_QUERY = "expected-identity";
    private static final int DEFAULT_CACHE_SIZE = 16;
    private final PackageManager mPackageManager;
    private final String mDigestAlgorithm;
    private final Cache mCache;
    private final Map<String, Map<String, Set<String>>> mPermissionAllowMap;
    private final Map<String, Set<String>> mExpectedIdentities;

    AppSignatureVerifier(Context context, Map<String, Map<String, Set<String>>> permissionAllowMap, Map<String, Set<String>> expectedIdentities, String digestAlgorithm, Cache cache) {
        this.mPackageManager = context.getPackageManager();
        this.mPermissionAllowMap = permissionAllowMap;
        this.mExpectedIdentities = expectedIdentities;
        this.mDigestAlgorithm = digestAlgorithm;
        this.mCache = cache;
    }

    static Builder builder(Context context) {
        return new Builder(context);
    }

    boolean verifySigningIdentity(String packageName, String permission) {
        Map<String, Set<String>> allowedCertDigests = this.mPermissionAllowMap.get(permission);
        if (allowedCertDigests == null) {
            Log.d((String)TAG, (String)("No expected signing identities declared for permission " + permission));
            return false;
        }
        Set<String> packageCertDigests = allowedCertDigests.get(packageName);
        Set<String> allPackagesCertDigests = allowedCertDigests.get("all-packages");
        if (packageCertDigests == null && allPackagesCertDigests == null) {
            return false;
        }
        return this.verifySigningIdentityForQuery(packageName, permission, packageCertDigests, allPackagesCertDigests);
    }

    boolean verifyExpectedIdentity(String packageName) {
        Set<String> packageCertDigests = this.mExpectedIdentities.get(packageName);
        if (packageCertDigests == null) {
            return false;
        }
        return this.verifySigningIdentityForQuery(packageName, EXPECTED_IDENTITY_QUERY, packageCertDigests, null);
    }

    boolean verifySigningIdentityForQuery(String packageName, String query, Set<String> packageCertDigests, Set<String> allPackagesCertDigests) {
        AppSigningInfo appSigningInfo;
        try {
            appSigningInfo = Build.VERSION.SDK_INT >= 28 ? Api28Impl.getAppSigningInfo(this.mPackageManager, packageName) : DefaultImpl.getAppSigningInfo(this.mPackageManager, packageName);
        }
        catch (AppSignatureVerifierException e) {
            Log.e((String)TAG, (String)("Caught an exception obtaining signing info for package " + packageName), (Throwable)e);
            return false;
        }
        CacheEntry cacheEntry = this.mCache.get(packageName, query);
        if (cacheEntry != null && cacheEntry.getLastUpdateTime() == appSigningInfo.getLastUpdateTime()) {
            return cacheEntry.getVerificationResult();
        }
        boolean verificationResult = appSigningInfo.getSigningLineage() != null ? this.verifySigningLineage(appSigningInfo.getSigningLineage(), packageCertDigests, allPackagesCertDigests) : this.verifyCurrentSigners(appSigningInfo.getCurrentSignatures(), packageCertDigests, allPackagesCertDigests);
        this.mCache.put(packageName, query, CacheEntry.create(verificationResult, appSigningInfo.getLastUpdateTime()));
        return verificationResult;
    }

    private boolean verifySigningLineage(List<Signature> signatures, Set<String> packageCertDigests, Set<String> allPackagesCertDigests) {
        for (Signature signature : signatures) {
            String signatureDigest = AppAuthenticatorUtils.computeDigest(this.mDigestAlgorithm, signature.toByteArray());
            if (packageCertDigests != null && packageCertDigests.contains(signatureDigest)) {
                return true;
            }
            if (allPackagesCertDigests == null || !allPackagesCertDigests.contains(signatureDigest)) continue;
            return true;
        }
        return false;
    }

    boolean verifyCurrentSigners(List<Signature> signatures, Set<String> packageCertDigests, Set<String> allPackagesCertDigests) {
        ArrayList<String> signatureDigests = new ArrayList<String>(signatures.size());
        for (Signature signature : signatures) {
            signatureDigests.add(AppAuthenticatorUtils.computeDigest(this.mDigestAlgorithm, signature.toByteArray()));
        }
        if (packageCertDigests != null && packageCertDigests.containsAll(signatureDigests)) {
            return true;
        }
        return allPackagesCertDigests != null && allPackagesCertDigests.containsAll(signatureDigests);
    }

    private static class AppSignatureVerifierException
    extends Exception {
        AppSignatureVerifierException(@NonNull String message) {
            super(message);
        }

        AppSignatureVerifierException(@NonNull String message, @NonNull Throwable cause) {
            super(message, cause);
        }
    }

    @AutoValue
    static abstract class AppSigningInfo {
        AppSigningInfo() {
        }

        abstract String getPackageName();

        abstract List<Signature> getCurrentSignatures();

        @Nullable
        abstract List<Signature> getSigningLineage();

        abstract long getLastUpdateTime();

        static AppSigningInfo create(@NonNull String packageName, @NonNull Signature[] currentSignatures, Signature[] signingLineage, long lastUpdateTime) {
            return new AutoValue_AppSignatureVerifier_AppSigningInfo(packageName, Arrays.asList(currentSignatures), signingLineage != null ? Arrays.asList(signingLineage) : null, lastUpdateTime);
        }
    }

    @AutoValue
    static abstract class CacheEntry {
        CacheEntry() {
        }

        abstract boolean getVerificationResult();

        abstract long getLastUpdateTime();

        static CacheEntry create(boolean verificationResult, long lastUpdateTime) {
            return new AutoValue_AppSignatureVerifier_CacheEntry(verificationResult, lastUpdateTime);
        }
    }

    static class Cache
    extends LruCache<String, CacheEntry> {
        Cache(int maxSize) {
            super(maxSize);
        }

        CacheEntry get(String packageName, String query) {
            return (CacheEntry)this.get(packageName + query);
        }

        void put(String packageName, String query, CacheEntry cacheEntry) {
            this.put(packageName + query, cacheEntry);
        }
    }

    private static class DefaultImpl {
        private DefaultImpl() {
        }

        @SuppressLint(value={"PackageManagerGetSignatures"})
        static AppSigningInfo getAppSigningInfo(PackageManager packageManager, String packageName) throws AppSignatureVerifierException {
            PackageInfo packageInfo;
            try {
                packageInfo = packageManager.getPackageInfo(packageName, 64);
            }
            catch (PackageManager.NameNotFoundException e) {
                throw new AppSignatureVerifierException("Package " + packageName + " not found", e);
            }
            if (packageInfo.signatures == null) {
                throw new AppSignatureVerifierException("No signatures returned for package " + packageName);
            }
            return AppSigningInfo.create(packageName, packageInfo.signatures, null, packageInfo.lastUpdateTime);
        }
    }

    @RequiresApi(value=28)
    private static class Api28Impl {
        private Api28Impl() {
        }

        static AppSigningInfo getAppSigningInfo(PackageManager packageManager, String packageName) throws AppSignatureVerifierException {
            PackageInfo packageInfo;
            try {
                packageInfo = packageManager.getPackageInfo(packageName, 0x8000000);
            }
            catch (PackageManager.NameNotFoundException e) {
                throw new AppSignatureVerifierException("Package " + packageName + " not found", e);
            }
            if (packageInfo.signingInfo == null) {
                throw new AppSignatureVerifierException("No SigningInfo returned for package " + packageName);
            }
            return AppSigningInfo.create(packageName, packageInfo.signingInfo.getApkContentsSigners(), packageInfo.signingInfo.getSigningCertificateHistory(), packageInfo.lastUpdateTime);
        }
    }

    static class Builder {
        private final Context mContext;
        private String mDigestAlgorithm;
        private Cache mCache;
        private Map<String, Map<String, Set<String>>> mPermissionAllowMap;
        private Map<String, Set<String>> mExpectedIdentities;

        Builder(Context context) {
            this.mContext = context;
        }

        Builder setDigestAlgorithm(String digestAlgorithm) {
            this.mDigestAlgorithm = digestAlgorithm;
            return this;
        }

        Builder setCache(Cache cache) {
            this.mCache = cache;
            return this;
        }

        Builder setPermissionAllowMap(Map<String, Map<String, Set<String>>> permissionAllowMap) {
            this.mPermissionAllowMap = permissionAllowMap;
            return this;
        }

        Builder setExpectedIdentities(Map<String, Set<String>> expectedIdentities) {
            this.mExpectedIdentities = expectedIdentities;
            return this;
        }

        AppSignatureVerifier build() {
            if (this.mPermissionAllowMap == null) {
                this.mPermissionAllowMap = new ArrayMap();
            }
            if (this.mExpectedIdentities == null) {
                this.mExpectedIdentities = new ArrayMap();
            }
            if (this.mDigestAlgorithm == null) {
                this.mDigestAlgorithm = "SHA-256";
            }
            if (this.mCache == null) {
                this.mCache = new Cache(16);
            }
            return new AppSignatureVerifier(this.mContext, this.mPermissionAllowMap, this.mExpectedIdentities, this.mDigestAlgorithm, this.mCache);
        }
    }
}

